# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['pyapp_flow',
 'pyapp_flow.ext',
 'pyapp_flow.ext.sphinx',
 'tests',
 'tests.unit',
 'tests.unit.ext']

package_data = \
{'': ['*']}

install_requires = \
['rich>=12.4.4,<13.0.0']

setup_kwargs = {
    'name': 'pyapp-flow',
    'version': '0.9.0',
    'description': 'Application workflow framework',
    'long_description': '# pyapp-flow\nA simple application level workflow library.\n\nAllows complex processes to be broken into smaller specific steps, greatly \nsimplifying testing and re-use.\n\n[![Quality Gate Status](https://sonarcloud.io/api/project_badges/measure?project=pyapp-org_pyapp-flow&metric=alert_status)](https://sonarcloud.io/summary/new_code?id=pyapp-org_pyapp-flow)\n[![Security Rating](https://sonarcloud.io/api/project_badges/measure?project=pyapp-org_pyapp-flow&metric=security_rating)](https://sonarcloud.io/summary/new_code?id=pyapp-org_pyapp-flow)\n[![Coverage](https://sonarcloud.io/api/project_badges/measure?project=pyapp-org_pyapp-flow&metric=coverage)](https://sonarcloud.io/summary/new_code?id=pyapp-org_pyapp-flow)\n[![Once you go Black...](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/ambv/black)\n\n\n## Installation\n\n```shell\npip install pyapp-flow\n```\n\n\n## Usage\n\n```python\nfrom pathlib import Path\nfrom typing import Sequence\nimport pyapp_flow as flow\n\n# Define steps:\n\n@flow.step(name="Load Names", output="names")\ndef load_names(root_path: Path) -> Sequence[str]:\n    """\n    Read a sequence of names from a file\n    """\n    with (root_path / "names.txt").open() as f_in:\n        return [name.strip() for name in f_in.readlines()]\n\n@flow.step(name="Say hello")\ndef say_hi(name: str):\n    print(f"Hello {name}")\n\n# Define a workflow:\n\ngreat_everybody = (\n    flow.Workflow(name="Great everybody in names file")\n    .nodes(\n      load_names,\n      flow.ForEach("name", in_var="names").loop(say_hi)\n    )\n)\n\n# Execute workflow:\n\ncontext = flow.WorkflowContext(root_path=Path())\ngreat_everybody(context)\n```\n\nAll nodes within the workflow follow a simple interface of:\n```python\ndef node_function(context: WorkflowContext):\n    ...\n```\nor using typing\n```python\nNodeFunction = Callable[[WorkflowContext], Any]\n```\n\nThe `step` decorator simplifies definition of a step by handling loading and saving \nof state variables from the `WorkflowContext`.\n\n\n## Reference\n\n### Workflow\n\nAt the basic level a workflow is an object that holds a series of nodes to be called \nin sequence. The workflow object also includes helper methods to generate and append\nthe nodes defined in the *Builtin Nodes* section of the documentation. \n\nJust like every node in pyApp-Flow a workflow is called with an `WorkflowContext` \nobject, this means workflows can be nested in workflows, or called from a for-each \nnode.\n\nThe one key aspect with a workflow object is related to context variable scope. \nWhen a workflow is triggered the context scope is copied and any changes made \nto the variables are discarded when the workflow ends. However, just like Python \nscoping only the reference to the variable is copied meaning mutable objects can \nbe modified (eg list/dicts).\n\n```python\nworkflow = (\n    Workflow(name="My Workflow")\n    .nodes(...)\n)\n```\n\n### WorkflowContext\n\nThe workflow context object holds the state of the workflow including handling \nvariable scoping and helper methods for logging progress.\n\n**Properties**\n\n- `state` \n\n  Direct access to state variables in the current scope.\n\n- `depth` \n \n  Current scope depth\n\n- `indent` \n\n  Helper that returns a string indent for use formatting messages\n\n**Methods**\n\n- `format`\n\n  Format a string using values from the context state. Most *name*\n  values for nodes/workflows use this method to allow values to be included\n  from scope eg:\n\n  ```python\n  context.format("Current path {working_path}")\n  ```\n\n- `push_state`/`pop_state`\n\n  Used to step into or out of a lower state scope. Typically these methods are\n  not called directly but are called via using a with block eg:\n  \n  ```python\n  with context:\n      pass  # Separate variable scope \n  ```\n\n- Logging wrappers\n\n  Wrappers around an internal workflow logger that handle indentation to make\n  reading the log easier.\n  \n  - log\n  - debug\n  - info\n  - warning\n  - error\n  - exception\n\n\n\n### Builtin Nodes\n\n**Modify context variables**\n\n- `SetVar`\n  \n    Set one or more variables into the context\n\n    ```python\n    SetVar(my_var="foo")\n    ```\n\n- `Append`\n\n    Append a value to a list in the context object (will create the list if it \n    does not exist).\n\n    ```python\n    Append("messages", "Operation failed to add {my_var}")\n    ```\n  \n- `CaptureErrors`\n\n    Capture and store any errors raised by node(s) within the capture block to a \n    variable within the context.\n\n    ```python\n    CaptureErrors("errors").nodes(my_flaky_step)\n    ```\n  \n    This node also has a `try_all` argument that controls the behaviour when an  \n    error is captured, if `True` every node is called even if they all raise errors,\n    this is useful for running a number of separate tests that may fail.\n\n    ```python\n    CaptureErrors(\n        "errors", \n        try_all=True\n    ).nodes(\n        my_first_check, \n        my_second_check, \n    )\n    ```\n\n**Provide feedback**\n\n- `LogMessage`\n    \n    Insert a message within optional values from the context into the runtime \n    log with an optional level.\n    \n    ```python\n    LogMessage("State of my_var is {my_var}", level=logging.INFO)\n    ```\n\n\n**Branching**\n\nBranching nodes utilise a fluent interface for defining the nodes within each \nbranch. \n\n- `Conditional` / `If`\n    \n    Analogous with an `if` statement, it can accept either a context variable \n    that can be interpreted as a `bool` or a function/lamba that accepts a \n    `WorkflowContext` object and returns a `bool`.\n\n    ```python \n    # With context variable\n    (\n        If("is_successful")\n        .true(log_message("Process successful :)"))\n        .false(log_message("Process failed :("))\n    )\n  \n    # With Lambda\n    (\n        If(lambda context: len(context.state.errors) == 0)\n        .true(log_message("Process successful :)"))\n        .false(log_message("Process failed :("))\n    )\n    ```\n  \n- `Switch`\n\n    Analogous with a `switch` statement found in many languages or with Python \n    a `dict` lookup with a default fallback.\n\n    Like the conditional node switch can accept a context variable or a \n    function/lambda that accepts a `WorkflowContext`, except returns any *hashable*\n    object.\n\n    ```python\n    # With context variable\n    (\n        Switch("my_var")\n        .case("foo", log_message("Found foo!"))\n        .case("bar", log_message("Found bar!"))\n        .default(log_message("Found neither."))\n    )\n  \n    # With Lambda\n    (\n        Switch(lambda context: context.state["my_var"])\n        .case("foo", log_message("Found foo!"))\n        .case("bar", log_message("Found bar!"))\n    )\n    ```\n  \n\n**Iteration**\n\n- `ForEach`\n    \n    Analogous with a `for` loop this node will iterate through a sequence and \n    call each of the child nodes.\n\n    All nodes within a for-each loop are in a nested context scope.\n    \n    ```python\n    # With a single target variable\n    (\n        ForEach("message", in_var="messages")\n        .loop(log_message("- {message}"))\n    )\n  \n    # With multiple target variables\n    (\n        ForEach("name, age", in_var="students")\n        .loop(log_message("- {name} is {age} years old."))\n    )\n    ```\n',
    'author': 'Tim Savage',
    'author_email': 'tim@savage.company',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/pyapp-org/pyapp-flow',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
