"""
    Copyright Engie Impact Sustainability Solution EMEAI 2020.
    All rights reserved.
"""

__author__ = 'Engie Impact Sustainability Solution EMEAI'

import json
import logging
from datetime import datetime
from decimal import Decimal

from eib_aws_utils.errors import EIBError, BackendError

_logger = logging.getLogger(__name__)


def http_endpoint(func):
    def inner1(*args, **kwargs):
        try:
            response = func(*args, **kwargs)

            if not isinstance(response, tuple):
                body, status = response, 200
            else:
                body, status = response

            return _to_api_response(body, status)

        except EIBError as error:
            _logger.log(error.log_level, f"[{error.title} ({error.http_status})] {error.detail}")
            return _to_api_response(error, error.http_status)

        except Exception as error:
            _logger.exception(f"An unhandled exception occurs: {error}")
            returned_exception = BackendError("Unhandled Exception")
            return _to_api_response(returned_exception, 500)

    return inner1


def _to_api_response(body, status):
    return dict(
        statusCode=str(status),
        body=json.dumps(body, default=_custom_class_dumper),
        headers={
            'Content-Type': 'application/json',
            'Access-Control-Allow-Origin': '*'
        }
    )


def _custom_class_dumper(o):
    if isinstance(o, datetime):
        return o.isoformat()

    elif isinstance(o, Decimal):
        return float(o)

    elif isinstance(o, EIBError):
        return dict(
            title=o.title,
            status=o.http_status,
            detail=o.detail
        )

    try:
        return o.to_dict()
    except AttributeError:
        pass

    raise TypeError(f"Object of type '{o.__class__.__name__}' is not JSON serializable")
