"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogClient = void 0;
const aws_lambda_shared_1 = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
/**
 * A client for working with the catalog.
 */
class CatalogClient {
    constructor() {
        this.bucketName = env_lambda_shared_1.requireEnv('CATALOG_BUCKET_NAME');
        this.objectKey = env_lambda_shared_1.requireEnv('CATALOG_OBJECT_KEY');
        this.s3 = aws_lambda_shared_1.s3();
    }
    /**
     * Creates a new client for accessing the catalog.
     */
    static async newClient() {
        const client = new CatalogClient();
        await client.init();
        return client;
    }
    /**
     * Downloads the catalog object and stores it in memory.
     */
    async init() {
        if (this._packages) {
            throw new Error('init() cannot be called twice');
        }
        this._packages = [];
        const params = {
            Bucket: this.bucketName,
            Key: this.objectKey,
        };
        let body;
        try {
            const data = await this.s3.getObject(params).promise();
            body = data.Body;
        }
        catch (e) {
            throw new Error(`No catalog was found at ${this.bucketName}/${this.objectKey}`);
        }
        if (!body) {
            throw new Error(`Catalog body is empty at ${this.bucketName}/${this.objectKey}`);
        }
        const contents = body.toString('utf-8');
        try {
            const data = JSON.parse(contents);
            if (typeof (data) != 'object') {
                throw new Error('Invalid format in catalog file. Expecting a map');
            }
            this._packages = data.packages;
        }
        catch (e) {
            throw new Error(`Unable to parse catalog file ${this.bucketName}/${this.objectKey}: ${e}`);
        }
    }
    /**
     * Returns a copy of the catalog list.
     */
    get packages() {
        if (!this._packages) {
            throw new Error('CatalogClient must be initialized');
        }
        return [...this._packages];
    }
}
exports.CatalogClient = CatalogClient;
//# sourceMappingURL=data:application/json;base64,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