"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inventory = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const canary_1 = require("./canary");
const constants_1 = require("./constants");
/**
 * Periodically computes an inventory of all indexed packages into the storage
 * bucket, and produces metrics with an overview of the index' state.
 */
class Inventory extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const rate = (_a = props.scheduleRate) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(5);
        this.canary = new canary_1.Canary(this, 'Resource', {
            description: '[ConstructHub/Inventory] A canary that periodically inspects the list of indexed packages',
            environment: { BUCKET_NAME: props.bucket.bucketName },
            logRetention: props.logRetention,
            memorySize: 10240,
            timeout: rate,
        });
        const grant = props.bucket.grantRead(this.canary);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(rate),
            targets: [new aws_events_targets_1.LambdaFunction(this.canary)],
        }).node.addDependency(grant);
        props.monitoring.watchful.watchLambdaFunction('Inventory Canary', this.canary);
    }
    metricMissingPackageMetadataCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    metricMissingAssemblyCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    metricMissingPythonDocsCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "MissingPythonDocsCount" /* MISSING_PYTHON_DOCS_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    metricMissingTypeScriptDocsCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "MissingPythonDocsCount" /* MISSING_PYTHON_DOCS_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    metricMissingPackageTarballCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    metricMissingSubmoduleCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "SubmoduleCount" /* SUBMODULE_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
    metricUnknownObjectCount(opts) {
        return new aws_cloudwatch_1.Metric({
            namespace: constants_1.METRICS_NAMESPACE,
            metricName: "UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */,
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
        });
    }
}
exports.Inventory = Inventory;
//# sourceMappingURL=data:application/json;base64,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