# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['python_fixturify_project']

package_data = \
{'': ['*']}

install_requires = \
['dict-path>=1.0.1,<2.0.0', 'wcmatch>=8.4.1,<9.0.0']

extras_require = \
{':python_version < "3.8"': ['importlib_metadata>=4.5.0,<5.0.0']}

setup_kwargs = {
    'name': 'python-fixturify-project',
    'version': '0.1.6',
    'description': 'Dynamic fixture creation for your tests',
    'long_description': '# python-fixturify-project\n\n[![Build status](https://github.com/scalvert/python-fixturify-project/workflows/build/badge.svg?branch=main&event=push)](https://github.com/scalvert/python-fixturify-project/actions?query=workflow%3Abuild)\n[![Python Version](https://img.shields.io/pypi/pyversions/python-fixturify-project.svg)](https://pypi.org/project/python-fixturify-project/)\n[![Dependencies Status](https://img.shields.io/badge/dependencies-up%20to%20date-brightgreen.svg)](https://github.com/scalvert/python-fixturify-project/pulls?utf8=%E2%9C%93&q=is%3Apr%20author%3Aapp%2Fdependabot)\n\n[![Code style: black](https://img.shields.io/badge/code%20style-black-000000.svg)](https://github.com/psf/black)\n[![Security: bandit](https://img.shields.io/badge/security-bandit-green.svg)](https://github.com/PyCQA/bandit)\n[![Pre-commit](https://img.shields.io/badge/pre--commit-enabled-brightgreen?logo=pre-commit&logoColor=white)](https://github.com/scalvert/python-fixturify-project/blob/master/.pre-commit-config.yaml)\n[![Semantic Versions](https://img.shields.io/badge/%20%20%F0%9F%93%A6%F0%9F%9A%80-semantic--versions-e10079.svg)](https://github.com/scalvert/python-fixturify-project/releases)\n[![License](https://img.shields.io/github/license/scalvert/python-fixturify-project)](https://github.com/scalvert/python-fixturify-project/blob/master/LICENSE)\n![Coverage Report](assets/images/coverage.svg)\n\n> Dynamic fixture creation for your tests\n\n_A Python port of [node-fixturify-project](https://github.com/stefanpenner/node-fixturify-project)_\n\n## Installation\n\n```bash\npip install -U python-fixturify-project\n```\n\nor install with `Poetry`\n\n```bash\npoetry add python-fixturify-project --dev\n```\n\n## Usage\n\n`python-fixturify-project` is a Python package that provides a way to create dynamic fixtures for your tests. Fixtures are real directories and files, written to a temporary directory.\n\n```python\nfrom python_fixturify_project import Project\n\ndir_json = {\n    "valid_file.txt": "some text",\n    "nested_dir": {\n        "valid_empty_file.txt": "",\n        "another_nested_empty_dir": {},\n        "another_nested_dir": {\n            "last_nested_empty_dir": {},\n            "final_text_file.txt": "some text",\n        },\n    },\n}\n\n# create a new project with the given directory structure\nwith Project(files=dir_json) as p:\n  # add new files to the project, merging with the existing directory structure\n  p.write({\n      "new_file.txt": "some text"\n  })\n\n  # read the actual contents on disc\n  actual_dir_json = p.read()\n```\n\n### Ignore Files\n\nBy default, the `read()` function will ignore all `.git` directories in your Project file structure. This can be overridden by using the `ignore_patterns` function parameter, which\ntakes a list of glob pattern strings. This may be slightly confusing, as glob patterns are normally used in an ***inclusive*** manner when performing file-system searches, however any patterns\nprovided to the `ignore_patterns` parameter will be used in an ***exclusive*** manner. For example:\n\n```python\nfiles = {\n    ".git": {\n        "a_nested_dir": {}\n    },\n    ".github": {\n        "ignore_me": {}, \n        "do_not_ignore_me": {\n            "a_file": "some text"\n        }\n    },\n    "ignore_me": "some text",\n    "do_not_ignore_me": "some text",\n}\n\nwith Project(files=files) as p:\n\n    dir_json = p.read(ignore_patterns=["**/.git", "**/.git/**", "**/ignore_me"])  # Default is ["**/.git", "**/.git/**"]\n\nassert dir_json == {\n    \'.github\': {\n        \'do_not_ignore_me\': {\n            \'a_file\': \'some text\',\n        },\n    },\n    \'do_not_ignore_me\': \'some text\',\n}\n```\n\n### Usage when writing tests\n\n`python-fixutrify-project` becomes even more useful when combining it with something like [`syrupy`](https://github.com/tophat/syrupy).\n\n```python\nfrom python_fixturify_project import Project\n\n\ndef test_mutating_project(snapshot):\n    with Project(files=INITIAL_DIR_JSON) as p:\n      mutate_files_for_some_reason(p.base_dir)\n\n      # ensure mutations were as expected\n      assert project.read() == snapshot\n```\n\n## Development\n\n### Makefile usage\n\n[`Makefile`](https://github.com/scalvert/python-fixturify-project/blob/main/Makefile) contains a lot of functions for faster development.\n\n<details>\n<summary>1. Download and remove Poetry</summary>\n<p>\n\nTo download and install Poetry run:\n\n```bash\nmake poetry-download\n```\n\nTo uninstall\n\n```bash\nmake poetry-remove\n```\n\n</p>\n</details>\n\n<details>\n<summary>2. Install all dependencies and pre-commit hooks</summary>\n<p>\n\nInstall requirements:\n\n```bash\nmake install\n```\n\nPre-commit hooks coulb be installed after `git init` via\n\n```bash\nmake pre-commit-install\n```\n\n</p>\n</details>\n\n<details>\n<summary>3. Codestyle</summary>\n<p>\n\nAutomatic formatting uses `pyupgrade`, `isort` and `black`.\n\n```bash\nmake codestyle\n\n# or use synonym\nmake formatting\n```\n\nCodestyle checks only, without rewriting files:\n\n```bash\nmake check-codestyle\n```\n\n> Note: `check-codestyle` uses `isort`, `black` and `darglint` library\n\nUpdate all dev libraries to the latest version using one comand\n\n```bash\nmake update-dev-deps\n```\n\n<details>\n<summary>4. Code security</summary>\n<p>\n\n```bash\nmake check-safety\n```\n\nThis command launches `Poetry` integrity checks as well as identifies security issues with `Safety` and `Bandit`.\n\n```bash\nmake check-safety\n```\n\n</p>\n</details>\n\n</p>\n</details>\n\n<details>\n<summary>5. Type checks</summary>\n<p>\n\nRun `mypy` static type checker\n\n```bash\nmake mypy\n```\n\n</p>\n</details>\n\n<details>\n<summary>6. Tests with coverage badges</summary>\n<p>\n\nRun `pytest`\n\n```bash\nmake test\n```\n\n</p>\n</details>\n\n<details>\n<summary>7. All linters</summary>\n<p>\n\nOf course there is a command to ~~rule~~ run all linters in one:\n\n```bash\nmake lint\n```\n\nthe same as:\n\n```bash\nmake test && make check-codestyle && make mypy && make check-safety\n```\n\n</p>\n</details>\n\n<details>\n<summary>8. Docker</summary>\n<p>\n\n```bash\nmake docker-build\n```\n\nwhich is equivalent to:\n\n```bash\nmake docker-build VERSION=latest\n```\n\nRemove docker image with\n\n```bash\nmake docker-remove\n```\n\nMore information [about docker](https://github.com/scalvert/python-fixturify-project/tree/main/docker).\n\n</p>\n</details>\n\n<details>\n<summary>9. Cleanup</summary>\n<p>\nDelete pycache files\n\n```bash\nmake pycache-remove\n```\n\nRemove package build\n\n```bash\nmake build-remove\n```\n\nDelete .DS_STORE files\n\n```bash\nmake dsstore-remove\n```\n\nRemove .mypycache\n\n```bash\nmake mypycache-remove\n```\n\nOr to remove all above run:\n\n```bash\nmake cleanup\n```\n\n</p>\n</details>\n\n## 🛡 License\n\n[![License](https://img.shields.io/github/license/scalvert/python-fixturify-project)](https://github.com/scalvert/python-fixturify-project/blob/master/LICENSE)\n\nThis project is licensed under the terms of the `MIT` license. See [LICENSE](https://github.com/scalvert/python-fixturify-project/blob/master/LICENSE) for more details.\n',
    'author': 'python-fixturify-project',
    'author_email': 'steve.calvert@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/scalvert/python-fixturify-project',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
