from genericpath import isdir
from os import makedirs
from pathlib import Path
from pythautomata.abstract.model_exporting_strategy import ModelExportingStrategy


class ImageExportingStrategy:

    '''
    Exports image of an automata generated by any exporting strategy based on Graphviz. 
    Works for DFA, Moore, Mealy and WFA

    ----------
    Attributes
    ----------
    exporter: ModelExportingStrategy
        Exporting strategy that will be used to generate the image.
    format: str
        Format of the image to be generated. It can be any format supported by Graphviz. (e.g., png, svg, pdf, etc.) 
    '''

    def __init__(self, exporting_strategy, format: str):
        self.exporter = exporting_strategy
        self.format = format

    def export(self, model, path):
        graph = self.exporter.create_graph(model)
        path = self.get_path_for(path, model)
        graph.render(path, cleanup=True, format=self.format)

    def get_path_for(self, path, model):
        if path is None:
            name = model.name
            path = "output_models/" + \
                ("" if name is None else f"{name}")
        if not isdir(path):
            makedirs(path)
        return Path(path, model.name)
