# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['xarray_dataclasses']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions>=3.10,<5.0']

extras_require = \
{':python_full_version >= "3.7.1" and python_version < "3.8"': ['numpy>=1.15,<1.22',
                                                                'xarray>=0.18,<0.21'],
 ':python_version >= "3.8" and python_version < "3.11"': ['numpy>=1.15,<2.0',
                                                          'xarray>=0.18,<2023']}

setup_kwargs = {
    'name': 'xarray-dataclasses',
    'version': '1.4.0',
    'description': 'xarray data creation made easy by dataclass',
    'long_description': '# xarray-dataclasses\n\n[![Release](https://img.shields.io/pypi/v/xarray-dataclasses?label=Release&color=cornflowerblue&style=flat-square)](https://pypi.org/project/xarray-dataclasses/)\n[![Python](https://img.shields.io/pypi/pyversions/xarray-dataclasses?label=Python&color=cornflowerblue&style=flat-square)](https://pypi.org/project/xarray-dataclasses/)\n[![Downloads](https://img.shields.io/pypi/dm/xarray-dataclasses?label=Downloads&color=cornflowerblue&style=flat-square)](https://pepy.tech/project/xarray-dataclasses)\n[![DOI](https://img.shields.io/badge/DOI-10.5281/zenodo.4624819-cornflowerblue?style=flat-square)](https://doi.org/10.5281/zenodo.4624819)\n[![Tests](https://img.shields.io/github/actions/workflow/status/astropenguin/xarray-dataclasses/tests.yml?label=Tests&style=flat-square)](https://github.com/astropenguin/xarray-dataclasses/actions)\n\nxarray data creation made easy by dataclass\n\n## Overview\n\nxarray-dataclasses is a Python package that makes it easy to create [xarray]\'s DataArray and Dataset objects that are "typed" (i.e. fixed dimensions, data type, coordinates, attributes, and name) using [the Python\'s dataclass]:\n\n```python\nfrom dataclasses import dataclass\nfrom typing import Literal\nfrom xarray_dataclasses import AsDataArray, Coord, Data\n\n\nX = Literal["x"]\nY = Literal["y"]\n\n\n@dataclass\nclass Image(AsDataArray):\n    """2D image as DataArray."""\n\n    data: Data[tuple[X, Y], float]\n    x: Coord[X, int] = 0\n    y: Coord[Y, int] = 0\n```\n\n### Features\n\n- Typed DataArray or Dataset objects can easily be created:\n  ```python\n  image = Image.new([[0, 1], [2, 3]], [0, 1], [0, 1])\n  ```\n- NumPy-like filled-data creation is also available:\n  ```python\n  image = Image.zeros([2, 2], x=[0, 1], y=[0, 1])\n  ```\n- Support for features by [the Python\'s dataclass] (`field`, `__post_init__`, ...).\n- Support for static type check by [Pyright].\n\n### Installation\n\n```shell\npip install xarray-dataclasses\n```\n\n## Basic usage\n\nxarray-dataclasses uses [the Python\'s dataclass].\nData (or data variables), coordinates, attributes, and a name of DataArray or Dataset objects will be defined as dataclass fields by special type hints (`Data`, `Coord`, `Attr`, `Name`), respectively.\nNote that the following code is supposed in the examples below.\n\n```python\nfrom dataclasses import dataclass\nfrom typing import Literal\nfrom xarray_dataclasses import AsDataArray, AsDataset\nfrom xarray_dataclasses import Attr, Coord, Data, Name\n\n\nX = Literal["x"]\nY = Literal["y"]\n```\n\n### Data field\n\nData field is a field whose value will become the data of a DataArray object or a data variable of a Dataset object.\nThe type hint `Data[TDims, TDtype]` fixes the dimensions and the data type of the object.\nHere are some examples of how to specify them.\n\nType hint | Inferred dimensions\n--- | ---\n`Data[tuple[()], ...]` | `()`\n`Data[Literal["x"], ...]` | `("x",)`\n`Data[tuple[Literal["x"]], ...]` | `("x",)`\n`Data[tuple[Literal["x"], Literal["y"]], ...]` | `("x", "y")`\n\nType hint | Inferred data type\n--- | ---\n`Data[..., Any]` | `None`\n`Data[..., None]` | `None`\n`Data[..., float]` | `numpy.dtype("float64")`\n`Data[..., numpy.float128]` | `numpy.dtype("float128")`\n`Data[..., Literal["datetime64[ns]"]]` | `numpy.dtype("<M8[ns]")`\n\n### Coordinate field\n\nCoordinate field is a field whose value will become a coordinate of a DataArray or a Dataset object.\nThe type hint `Coord[TDims, TDtype]` fixes the dimensions and the data type of the object.\n\n### Attribute field\n\nAttribute field is a field whose value will become an attribute of a DataArray or a Dataset object.\nThe type hint `Attr[TAttr]` specifies the type of the value, which is used only for static type check.\n\n### Name field\n\nName field is a field whose value will become the name of a DataArray object.\nThe type hint `Name[TName]` specifies the type of the value, which is used only for static type check.\n\n### DataArray class\n\nDataArray class is a dataclass that defines typed DataArray specifications.\nExactly one data field is allowed in a DataArray class.\nThe second and subsequent data fields are just ignored in DataArray creation.\n\n```python\n@dataclass\nclass Image(AsDataArray):\n    """2D image as DataArray."""\n\n    data: Data[tuple[X, Y], float]\n    x: Coord[X, int] = 0\n    y: Coord[Y, int] = 0\n    units: Attr[str] = "cd / m^2"\n    name: Name[str] = "luminance"\n```\n\nA DataArray object will be created by a class method `new()`:\n\n```python\nImage.new([[0, 1], [2, 3]], x=[0, 1], y=[0, 1])\n\n<xarray.DataArray "luminance" (x: 2, y: 2)>\narray([[0., 1.],\n       [2., 3.]])\nCoordinates:\n  * x        (x) int64 0 1\n  * y        (y) int64 0 1\nAttributes:\n    units:    cd / m^2\n```\n\nNumPy-like class methods (`zeros()`, `ones()`, ...) are also available:\n\n```python\nImage.ones((3, 3))\n\n<xarray.DataArray "luminance" (x: 3, y: 3)>\narray([[1., 1., 1.],\n       [1., 1., 1.],\n       [1., 1., 1.]])\nCoordinates:\n  * x        (x) int64 0 0 0\n  * y        (y) int64 0 0 0\nAttributes:\n    units:    cd / m^2\n```\n\n### Dataset class\n\nDataset class is a dataclass that defines typed Dataset specifications.\nMultiple data fields are allowed to define the data variables of the object.\n\n```python\n@dataclass\nclass ColorImage(AsDataset):\n    """2D color image as Dataset."""\n\n    red: Data[tuple[X, Y], float]\n    green: Data[tuple[X, Y], float]\n    blue: Data[tuple[X, Y], float]\n    x: Coord[X, int] = 0\n    y: Coord[Y, int] = 0\n    units: Attr[str] = "cd / m^2"\n```\n\nA Dataset object will be created by a class method `new()`:\n\n```python\nColorImage.new(\n    [[0, 0], [0, 0]],  # red\n    [[1, 1], [1, 1]],  # green\n    [[2, 2], [2, 2]],  # blue\n)\n\n<xarray.Dataset>\nDimensions:  (x: 2, y: 2)\nCoordinates:\n  * x        (x) int64 0 0\n  * y        (y) int64 0 0\nData variables:\n    red      (x, y) float64 0.0 0.0 0.0 0.0\n    green    (x, y) float64 1.0 1.0 1.0 1.0\n    blue     (x, y) float64 2.0 2.0 2.0 2.0\nAttributes:\n    units:    cd / m^2\n```\n\n## Advanced usage\n\n### Coordof and Dataof type hints\n\nxarray-dataclasses provides advanced type hints, `Coordof` and `Dataof`.\nUnlike `Data` and `Coord`, they specify a dataclass that defines a DataArray class.\nThis is useful when users want to add metadata to dimensions for [plotting].\nFor example:\n\n```python\nfrom xarray_dataclasses import Coordof\n\n\n@dataclass\nclass XAxis:\n    data: Data[X, int]\n    long_name: Attr[str] = "x axis"\n    units: Attr[str] = "pixel"\n\n\n@dataclass\nclass YAxis:\n    data: Data[Y, int]\n    long_name: Attr[str] = "y axis"\n    units: Attr[str] = "pixel"\n\n\n@dataclass\nclass Image(AsDataArray):\n    """2D image as DataArray."""\n\n    data: Data[tuple[X, Y], float]\n    x: Coordof[XAxis] = 0\n    y: Coordof[YAxis] = 0\n```\n\n### General data variable names in Dataset creation\n\nDue to the limitation of Python\'s parameter names, it is not possible to define data variable names that contain white spaces, for example.\nIn such cases, please define DataArray classes of each data variable so that they have name fields and specify them by `Dataof` in a Dataset class.\nThen the values of the name fields will be used as data variable names.\nFor example:\n\n```python\n@dataclass\nclass Red:\n    data: Data[tuple[X, Y], float]\n    name: Name[str] = "Red image"\n\n\n@dataclass\nclass Green:\n    data: Data[tuple[X, Y], float]\n    name: Name[str] = "Green image"\n\n\n@dataclass\nclass Blue:\n    data: Data[tuple[X, Y], float]\n    name: Name[str] = "Blue image"\n\n\n@dataclass\nclass ColorImage(AsDataset):\n    """2D color image as Dataset."""\n\n    red: Dataof[Red]\n    green: Dataof[Green]\n    blue: Dataof[Blue]\n```\n\n```python\nColorImage.new(\n    [[0, 0], [0, 0]],\n    [[1, 1], [1, 1]],\n    [[2, 2], [2, 2]],\n)\n\n<xarray.Dataset>\nDimensions:      (x: 2, y: 2)\nDimensions without coordinates: x, y\nData variables:\n    Red image    (x, y) float64 0.0 0.0 0.0 0.0\n    Green image  (x, y) float64 1.0 1.0 1.0 1.0\n    Blue image   (x, y) float64 2.0 2.0 2.0 2.0\n```\n\n### Customization of DataArray or Dataset creation\n\nFor customization, users can add a special class attribute, `__dataoptions__`, to a DataArray or Dataset class.\nA custom factory for DataArray or Dataset creation is only supported in the current implementation.\n\n\n```python\nimport xarray as xr\nfrom xarray_dataclasses import DataOptions\n\n\nclass Custom(xr.DataArray):\n    """Custom DataArray."""\n\n    __slots__ = ()\n\n    def custom_method(self) -> bool:\n        """Custom method."""\n        return True\n\n\n@dataclass\nclass Image(AsDataArray):\n    """2D image as DataArray."""\n\n    data: Data[tuple[X, Y], float]\n    x: Coord[X, int] = 0\n    y: Coord[Y, int] = 0\n\n    __dataoptions__ = DataOptions(Custom)\n\n\nimage = Image.ones([3, 3])\nisinstance(image, Custom)  # True\nimage.custom_method()  # True\n```\n\n### DataArray and Dataset creation without shorthands\n\nxarray-dataclasses provides functions, `asdataarray` and `asdataset`.\nThis is useful when users do not want to inherit the mix-in class (`AsDataArray` or `AsDataset`) in a DataArray or Dataset dataclass.\nFor example:\n\n```python\nfrom xarray_dataclasses import asdataarray\n\n\n@dataclass\nclass Image:\n    """2D image as DataArray."""\n\n    data: Data[tuple[X, Y], float]\n    x: Coord[X, int] = 0\n    y: Coord[Y, int] = 0\n\n\nimage = asdataarray(Image([[0, 1], [2, 3]], [0, 1], [0, 1]))\n```\n\n\n<!-- References -->\n[Pyright]: https://github.com/microsoft/pyright\n[the Python\'s dataclass]: https://docs.python.org/3/library/dataclasses.html\n[xarray]: https://xarray.pydata.org/en/stable/index.html\n[plotting]: https://xarray.pydata.org/en/stable/user-guide/plotting.html#simple-example\n',
    'author': 'Akio Taniguchi',
    'author_email': 'taniguchi@a.phys.nagoya-u.ac.jp',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/astropenguin/xarray-dataclasses/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.7.1,<3.11',
}


setup(**setup_kwargs)
