import re
from datetime import datetime
from regexapp import LinePattern
from regexapp import MultilinePattern
from regexapp.exceptions import RegexBuilderError
from regexapp.exceptions import PatternReferenceError
from regexapp.collection import REF
import regexapp
from copy import deepcopy
from collections import OrderedDict
from textwrap import indent
from textwrap import dedent

BASELINE_REF = deepcopy(REF)


def is_pro_edition():
    """return True if regexapp is Pro or Enterprise edition"""
    chk = regexapp.edition == 'Pro' or regexapp.edition == 'Enterprise'
    return chk


def enclose_string(text):
    """enclose text with either double-quote or triple double-quote

    Parameters
    ----------
    text (str): a text

    Returns
    -------
    str: a new string with enclosed double-quote or triple double-quote
    """
    text = str(text)
    fmt = '"""{}"""' if len(text.splitlines()) > 1 else '"{}"'
    enclosed_txt = fmt.format(text.replace('"', r'\"'))
    return enclosed_txt


def create_custom_docstring(**kwargs):
    """Generate custom docstring for Pro Edition or Enterprise Edition

    Parameters
    ----------
    kwargs (dict): custom keyword arguments for
            Pro Edition or Enterprise Edition.

    Returns
    -------
    str: a custom docstring or empty string.
    """
    if kwargs:
        if is_pro_edition():
            fmt = ('Contact tuyen@geekstrident.com to use {!r} flags on '
                   'Regexapp Pro or Enterprise Edition.')
            print(fmt.format(kwargs))
        else:
            fmt = 'CANT use {!r} flags with Regexapp Community Edition.'
            print(fmt.format(kwargs))
            return ''
    else:
        return ''


def create_docstring(test_framework='unittest',
                     author='', email='', company='', **kwargs):
    """Generate module docstring for test script

    Parameters
    ----------
    test_framework (str): a test framework.  Default is unittest.
    author (str): author name.  Default is empty.
    email (str): author name.  Default is empty.
    company (str): company name.  Default is empty.
    kwargs (dict): custom keyword arguments for
            Pro Edition or Enterprise Edition.

    Returns
    -------
    str: a module docstring
    """
    lang = '' if test_framework == 'robotframework' else 'Python '
    fmt = '{}{} script is generated by Regexapp {} Edition'
    fmt1 = 'Created by  : {}'
    fmt2 = 'Email       : {}'
    fmt3 = 'Company     : {}'
    fmt4 = 'Created date: {:%Y-%m-%d}'

    lst = list()
    author = author or company
    lst.append(fmt.format(lang, test_framework, regexapp.edition))
    lst.append('')
    author and lst.append(fmt1.format(author))
    email and lst.append(fmt2.format(email))
    company and company != author and lst.append(fmt3.format(company))
    lst.append(fmt4.format(datetime.now()))
    custom_docstr = create_custom_docstring(**kwargs)
    custom_docstr and lst.append(custom_docstr)

    if test_framework == 'robotframework':
        new_lst = [line.strip() for line in indent('\n'.join(lst), '# ').splitlines()]
        comment = '\n'.join(new_lst)
        return comment
    else:
        module_docstr = '"""{}\n"""'.format('\n'.join(lst))
        return module_docstr


def save_file(filename, content):
    """Save data to file

    Parameters
    ----------
    filename (str): a file name
    content (str): a file content
    """
    filename = str(filename).strip()
    if filename:
        with open(filename, 'w') as stream:
            stream.write(content)


class RegexBuilder:
    """Use for building regex pattern

    Attributes
    ----------
    user_data (str, list): a user data can be either string or list of string.
    test_data (str, list): a test data can be either string or list of string.

    prepended_ws (bool): prepend a whitespace at the beginning of a pattern.
            Default is False.
    appended_ws (bool): append a whitespace at the end of a pattern.
            Default is False.
    ignore_case (bool): prepend (?i) at the beginning of a pattern.
            Default is False.

    is_line (bool): a flag to use LinePattern.  Default is False.
    test_name (str): a predefined test name.  Default is empty.
            + unittest will use either predefined test name or
                generated test name from test data
            + pytest will use predefined test name.
            + robotframework test will depend on test workflow.  It might be
                either used predefined test name or generated test name.
    max_words (int): total number of words for generating test name.
            Default is 6 words.
    test_cls_name (str): a test class name for test script.  This test class
            name only be applicable for unittest or pytest.
            Default is TestDynamicGenTestScript.

    author (str): author name.  Default is empty.
    email (str): author name.  Default is empty.
    company (str): company name.  Default is empty.

    filename (str): save a generated test script to file name.
    kwargs (dict): an optional keyword arguments.
            Community edition will use the following keywords:
                prepended_ws, appended_ws, ignore_case
            Pro or Enterprise edition will use
                prepended_ws, appended_ws, ignore_case, other keywords

    patterns (list): a list of patterns.
    test_report (str): a test report.
    test_result (bool): a test result.
    user_data_pattern_table (OrderedDict): a variable holds (user_data, pattern) pair.
    pattern_user_data_table (OrderedDict): a variable holds (pattern, user_data) pair.
    test_data_pattern_table (OrderedDict): a variable holds (test_data, pattern) pair.
    pattern_test_data_table (OrderedDict): a variable holds (pattern, test_data) pair.

    Methods
    -------
    RegexBuilder.validate_data(data, name) -> bool
    build() -> None
    test(showed=True) -> bool
    create_unittest() -> str
    create_pytest() -> str
    create_rf_test() -> str
    create_python_test() -> str

    Raises
    ------
    RegexBuilderError: if user_data or test_data is invalid format.
    """
    def __init__(self, user_data='', test_data='',
                 prepended_ws=False, appended_ws=False, ignore_case=False,
                 test_name='', is_line=False,
                 max_words=6, test_cls_name='TestDynamicGenTestScript',
                 author='', email='', company='', filename='',
                 **kwargs
                 ):
        self.user_data = user_data
        self.test_data = test_data

        self.prepended_ws = prepended_ws
        self.appended_ws = appended_ws
        self.ignore_case = ignore_case

        self.test_name = test_name
        self.is_line = is_line
        self.max_words = max_words
        self.test_cls_name = test_cls_name
        self.author = author
        self.email = email
        self.company = company
        self.filename = filename
        self.kwargs = kwargs

        self.patterns = []
        self.test_report = ''
        self.test_result = False
        self.user_data_pattern_table = OrderedDict()    # user data via pattern
        self.pattern_user_data_table = OrderedDict()    # pattern via user data
        self.test_data_pattern_table = OrderedDict()    # test data via pattern
        self.pattern_test_data_table = OrderedDict()    # pattern via test data

    @classmethod
    def validate_data(cls, **kwargs):
        """validate data

        Parameters
        ----------
        kwargs (dict): keyword argument

        Returns
        -------
        bool: True or False

        Raises
        ------
        RegexBuilderError: if failed to validate data.
        """
        if not kwargs:
            msg = 'CANT validate data without providing data.'
            raise RegexBuilderError(msg)

        is_validated = True
        for name, data in kwargs.items():
            fmt = '{} MUST be string or list of string.'
            if not isinstance(data, (list, str)):
                msg = fmt.format(name)
                raise RegexBuilderError(msg)

            if isinstance(data, list):
                for line in data:
                    if not isinstance(line, str):
                        msg = fmt.format(name)
                        raise RegexBuilderError(msg)
            is_validated &= True if data else False
        return is_validated

    def build(self):
        """Build regex pattern"""
        data = self.user_data
        self.__class__.validate_data(user_data=data)

        if not data:
            self.test_report = 'CANT build regex pattern with an empty data.'
            print(self.test_report)
            return

        if self.is_line:
            lst_of_user_data = data[:] if isinstance(data, (list, tuple)) else data.splitlines()
        else:
            if isinstance(data, str):
                lst_of_user_data = [data]
            else:
                lst_of_user_data = []
                for item in data:
                    if isinstance(item, (list, tuple)):
                        lst_of_user_data.append('\n'.join(map(str, item)))
                    else:
                        lst_of_user_data.append(str(item))

        for user_data in lst_of_user_data:
            if self.is_line:
                pattern = LinePattern(
                    user_data,
                    prepended_ws=self.prepended_ws,
                    appended_ws=self.appended_ws,
                    ignore_case=self.ignore_case
                )
            else:
                pattern = MultilinePattern(user_data, ignore_case=self.ignore_case)

            pattern not in self.patterns and self.patterns.append(pattern)
            self.user_data_pattern_table[user_data] = pattern
            self.pattern_user_data_table[pattern] = user_data

    def test(self, showed=False):
        """test regex pattern via test data.

        Parameters
        ----------
        showed (bool): show test report if set to True.  Default is False.

        Returns
        -------
        bool: True if passed a test, otherwise, False.
        """
        data = self.test_data
        self.__class__.validate_data(test_data=data)

        if not data:
            self.test_report = 'CANT run test with an empty data.'
            showed and print(self.test_report)
            return False

        if self.is_line:
            lst_of_test_data = data[:] if isinstance(data, (list, tuple)) else data.splitlines()
        else:
            if isinstance(data, str):
                lst_of_test_data = [data]
            else:
                lst_of_test_data = []
                for item in data:
                    if isinstance(item, (list, tuple)):
                        lst_of_test_data.append('\n'.join(map(str, item)))
                    else:
                        lst_of_test_data.append(str(item))

        result = ['Test Data:', '-' * 9, '\n'.join(lst_of_test_data), '']
        result += ['Matched Result:', '-' * 14]

        test_result = True
        for pat in self.patterns:
            is_matched = False
            lst = []
            for test_data in lst_of_test_data:
                match = re.search(pat, test_data)
                if match:
                    is_matched = True
                    match.groupdict() and lst.append(match.groupdict())
                    self.test_data_pattern_table[test_data] = pat
                    self.pattern_test_data_table[pat] = test_data

            test_result &= is_matched
            tr = 'NO' if not is_matched else lst if lst else 'YES'
            result.append('pattern: {}'.format(pat))
            result.append('matched: {}'.format(tr))
            result.append('-' * 10)

        self.test_result = test_result
        self.test_report = '\n'.join(result)
        showed and print(self.test_report)

        return test_result

    def create_unittest(self):
        """dynamically generate Python unittest script

        Returns
        -------
        str: python unittest script
        """
        factory = DynamicTestScriptBuilder(test_info=self)
        script = factory.create_unittest()
        return script

    def create_pytest(self):
        """dynamically generate Python pytest script

        Returns
        -------
        str: python pytest script
        """
        factory = DynamicTestScriptBuilder(test_info=self)
        script = factory.create_pytest()
        return script

    def create_rf_test(self):
        """dynamically generate Robotframework script

        Returns
        -------
        str: Robotframework test script
        """
        factory = DynamicTestScriptBuilder(test_info=self)
        script = factory.create_rf_test()
        return script

    def create_python_test(self):
        """dynamically generate Python test script

        Returns
        -------
        str: python test script
        """
        factory = DynamicTestScriptBuilder(test_info=self)
        script = factory.create_python_test()
        return script


def add_reference(name='', pattern='', **kwargs):
    """add keyword reference to PatternReference.  This is an inline adding
    PatternReference for quick test.

    Parameters
    ----------
    name (str): a keyword.
    pattern (str): a regex pattern.
    kwargs (dict): keyword argument which will use for special case such as datetime.

    Raises
    ------
    PatternReferenceError: if adding an existing keyword from
        system_references.yaml or user_references.yaml
    """
    if not name:
        fmt = '{} keyword can not be empty name.'
        PatternReferenceError(fmt.format(name))

    obj = dict(pattern=pattern,
               description='inline_{}_{}'.format(name, pattern))
    if name not in REF:
        REF[name] = obj
    else:
        if name == 'datetime':
            for key, value in kwargs.items():
                if re.match(r'format\d+$', key):
                    REF['datetime'][key] = value
        else:
            if name not in BASELINE_REF:
                REF[name] = obj
            else:
                fmt = ('{} already exists in system_references.yaml '
                       'or user_references.yaml')
                raise PatternReferenceError(fmt.format(name))


def remove_reference(name=''):
    """remove keyword reference from PatternReference.  This method only remove
    any inline adding keyword reference.

    Parameters
    ----------
    name (str): a keyword.

    Raises
    ------
    PatternReferenceError: if removing an existing keyword from
        system_references.yaml or user_references.yaml
    """
    if not name:
        fmt = '{} keyword can not be empty name.'
        PatternReferenceError(fmt.format(name))

    if name in REF:
        if name not in BASELINE_REF:
            REF.pop(name)
        else:
            if name == 'datetime':
                REF['datetime'] = deepcopy(BASELINE_REF['datetime'])
            else:
                fmt = ('CANT remove {!r} from system_references.yaml '
                       'or user_references.yaml')
                raise PatternReferenceError(fmt.format(name))
    else:
        fmt = 'CANT remove {!r} keyword because it does not exist.'
        raise PatternReferenceError(fmt.format(name))


class DynamicTestScriptBuilder:
    """Dynamically generate test script

    Attributes
    ----------
    test_info (RegexBuilder, list): can be either RegexBuilder instance or
            a list of string.  If test_info is a list of string, it must follow
            this format that contain two items: user_data and test_data.
    test_name (str): a predefined test name.  Default is empty.
            + unittest will use either predefined test name or
                generated test name from test data
            + pytest will use predefined test name.
            + robotframework test will depend on test workflow.  It might be
                either used predefined test name or generated test name.
    is_line (bool): a flag to use LinePattern.  Default is False.
    max_words (int): total number of words for generating test name.
            Default is 6 words.
    test_cls_name (str): a test class name for test script.  This test class
            name only be applicable for unittest or pytest.
            Default is TestDynamicGenTestScript.
    author (str): author name.  Default is empty.
    email (str): author name.  Default is empty.
    company (str): company name.  Default is empty.
    filename (str): save a generated test script to file name.
    kwargs (dict): an optional keyword arguments.
            Community edition will use the following keywords:
                prepended_ws, appended_ws, ignore_case
            Pro or Enterprise edition will use
                prepended_ws, appended_ws, ignore_case, other keywords

    lst_of_tests (list): a list of test.
    test_data (str): a test data
    patterns (list): a list of patterns.

    Methods
    -------
    compile_test_info() -> None
    generate_test_name(test_data='') -> str
    create_unittest() -> str
    create_pytest() -> str
    create_rf_test() -> str
    create_python_test() -> str
    """
    def __init__(self, test_info=None, test_name='', is_line=False,
                 max_words=6, test_cls_name='TestDynamicGenTestScript',
                 author='', email='', company='', filename='',
                 **kwargs):
        self.test_info = test_info
        if isinstance(test_info, RegexBuilder):
            self.test_name = test_info.test_name
            self.is_line = is_line or test_info.is_line
            self.max_words = test_info.max_words
            self.test_cls_name = test_info.test_cls_name

            self.author = author or test_info.author
            self.email = email or test_info.email
            self.company = company or test_info.company
            self.filename = filename or test_info.filename

            self.prepended_ws = test_info.prepended_ws
            self.appended_ws = test_info.appended_ws
            self.ignore_case = test_info.ignore_case
            self.kwargs = test_info.kwargs
        else:
            self.test_name = test_name
            self.is_line = is_line
            self.max_words = max_words
            self.test_cls_name = test_cls_name

            self.author = author
            self.email = email
            self.company = company
            self.filename = filename

            self.prepended_ws = kwargs.get('prepended_ws', False)
            self.appended_ws = kwargs.get('appended_ws', False)
            self.ignore_case = kwargs.get('ignore_case', False)
            self.kwargs = kwargs

        'prepended_ws' in self.kwargs and self.kwargs.pop('prepended_ws')
        'appended_ws' in self.kwargs and self.kwargs.pop('appended_ws')
        'ignore_case' in self.kwargs and self.kwargs.pop('ignore_case')

        self.lst_of_tests = []
        self.test_data = None
        self.patterns = None
        self.compile_test_info()

    def compile_test_info(self):
        """prepare a list of test cases from test info"""

        self.lst_of_tests = []
        test_info = self.test_info
        if isinstance(test_info, RegexBuilder):
            testable = deepcopy(test_info)
            self.test_data = testable.test_data
        else:
            chk = isinstance(test_info, list) and len(test_info) == 2
            if not chk:
                raise RegexBuilderError('Invalid test_info format')

            user_data, test_data = self.test_info
            self.test_data = test_data

            testable = RegexBuilder(
                user_data=user_data, test_data=test_data,
                is_line=self.is_line,
                prepended_ws=self.prepended_ws,
                appended_ws=self.appended_ws,
                ignore_case=self.ignore_case
            )
        testable.build()
        testable.test()

        self.patterns = testable.patterns

        user_data_pattern_table = testable.user_data_pattern_table
        pattern_user_data_table = testable.pattern_user_data_table
        test_data_pattern_table = testable.test_data_pattern_table

        if not test_data_pattern_table and not user_data_pattern_table:
            raise RegexBuilderError('No prepared_data to build test script')

        for test_data, pattern in test_data_pattern_table.items():
            test_name = self.generate_test_name(test_data=test_data)
            prepared_data = pattern_user_data_table.get(pattern)
            self.lst_of_tests.append([test_name, test_data, prepared_data, pattern])

    def generate_test_name(self, test_data=''):
        """generate test name from test_data

        Parameters
        ----------
        test_data (str): a test data.  Default is empty.

        Returns
        -------
        str: a test name
        """
        pat = r'[^0-9a-zA-Z]*\s+[^0-9a-zA-Z]*'
        test_data = str(test_data).lower().strip()
        test_data = ' '.join(re.split(pat, test_data)[:self.max_words])
        test_name = self.test_name or test_data
        test_name = re.sub(r'[^0-9a-z]+', '_', test_name)
        test_name = test_name.strip('_')
        if not test_name.startswith('test_'):
            test_name = 'test_{}'.format(test_name)
        return test_name

    def create_unittest(self):
        """dynamically generate Python unittest script

        Returns
        -------
        str: python unittest script
        """

        factory = UnittestBuilder(self)
        test_script = factory.create()
        save_file(self.filename, test_script)
        return test_script

    def create_pytest(self):
        """dynamically generate Python pytest script
        Returns
        -------
        str: python pytest script
        """

        factory = PytestBuilder(self)
        test_script = factory.create()
        save_file(self.filename, test_script)
        return test_script

    def create_rf_test(self):     # noqa
        """dynamically generate Robotframework test script

        Returns
        -------
        str: Robotframework test script
        """
        msg = 'TODO: need to implement generated_rf_test for robotframework'
        NotImplementedError(msg)

    def create_python_test(self):
        """dynamically generate Python test script

        Returns
        -------
        str: a Python test script
        """
        factory = PythonSnippetBuilder(self)
        test_script = factory.create()
        save_file(self.filename, test_script)
        return test_script


class UnittestBuilder:
    """Create Unittest script

    Attributes
    ----------
    tc_gen (DynamicGenTestScript): an DynamicGenTestScript instance.
    module_docstring (str): a Python snippet docstring.

    Methods
    -------
    create_testcase_class() -> str
    create_testcase_method() -> str
    create_testcase_load() -> str
    create() -> str
    """
    def __init__(self, tc_gen):
        self.tc_gen = tc_gen
        self.module_docstring = create_docstring(
            test_framework='unittest',
            author=tc_gen.author,
            email=tc_gen.email,
            company=tc_gen.company,
            **tc_gen.kwargs
        )

    def create_testcase_class(self):
        """return partial unit test class definition"""

        tmpl = """
          {module_docstring}

          import unittest
          import re

          class {test_cls_name}(unittest.TestCase):
              def __init__(self, test_name='', test_data=None, pattern=None):
                  super().__init__(test_name)
                  self.test_data = test_data
                  self.pattern = pattern
        """
        tmpl = dedent(tmpl).strip()

        tc_gen = self.tc_gen
        partial_script = tmpl.format(
            module_docstring=self.module_docstring,
            test_cls_name=tc_gen.test_cls_name
        )

        return partial_script

    def create_testcase_method(self):
        """return partial unit test method definition"""

        tmpl = """
            def {test_name}(self):
                result = re.search(self.pattern, self.test_data)
                self.assertIsNotNone(result)
        """
        tmpl = dedent(tmpl).strip()

        tc_gen = self.tc_gen
        lst = []

        for test in tc_gen.lst_of_tests:
            test_name = test[0]
            method_def = tmpl.format(test_name=test_name)
            method_def = indent(method_def, ' ' * 4)
            if method_def not in lst:
                lst.append(method_def)
                lst.append('')

        partial_script = '\n'.join(lst)
        return partial_script

    def create_testcase_load(self):
        """return partial unit test load_tests function definition"""

        tmpl = """
            def load_tests(loader, tests, pattern):
                test_cases = unittest.TestSuite()
                
                {data_insertion}
                
                for arg in arguments:
                    test_name, test_data, pattern = arg
                    testcase = {test_cls_name}(
                        test_name=test_name,
                        test_data=test_data,
                        pattern=pattern
                    )
                    test_cases.addTest(testcase)
                return test_cases
        """
        tmpl = dedent(tmpl).strip()

        tmpl_data = """
            arguments.append(
                (
                    {test_name},    # test name
                    {test_data},    # test data
                    r{pattern}   # pattern
                )
            )
        """
        tmpl_data = dedent(tmpl_data).strip()

        tc_gen = self.tc_gen
        lst = ['arguments = list()']

        test_desc_fmt = '    # test case #{:0{}} - {}'
        spacers = len(str(len(tc_gen.lst_of_tests)))
        for index, test in enumerate(tc_gen.lst_of_tests, 1):
            test_name, test_data, _, pattern = test
            test_data = enclose_string(test_data)
            data = tmpl_data.format(
                test_name=enclose_string(test_name),
                test_data='__test_data_placeholder__',
                pattern=enclose_string(pattern))
            lst.append('')
            lst.append(test_desc_fmt.format(index, spacers, test_name))
            new_data = indent(data, ' ' * 4)
            new_data = new_data.replace('__test_data_placeholder__', test_data)
            lst.append(new_data)

        data_insertion = '\n'.join(lst)
        sub_script = tmpl.format(
            data_insertion=data_insertion,
            test_cls_name=tc_gen.test_cls_name
        )
        return sub_script

    def create(self):
        """return Python unittest script"""
        tc_class = self.create_testcase_class()
        tc_method = self.create_testcase_method()
        tc_load = self.create_testcase_load()

        test_script = '{}\n\n{}\n\n{}'.format(tc_class, tc_method, tc_load)
        return test_script


class PytestBuilder:
    """Create Pytest script

    Attributes
    ----------
    tc_gen (DynamicGenTestScript): an DynamicGenTestScript instance.
    module_docstring (str): a Python snippet docstring.

    Methods
    -------
    create() -> str
    """
    def __init__(self, tc_gen):
        self.tc_gen = tc_gen
        self.module_docstring = create_docstring(
            test_framework='pytest',
            author=tc_gen.author,
            email=tc_gen.email,
            company=tc_gen.company,
            **tc_gen.kwargs
        )

    def create(self):
        """return pytest script"""

        tmpl = """
            {module_docstring}
            
            import pytest
            import re
            
            class {test_cls_name}:
                @pytest.mark.parametrize(
                    ('test_data', 'pattern'),
                    (
                        {parametrize_data}
                    )
                )
                def {test_name}(self, test_data, pattern):
                    result = re.search(pattern, test_data)
                    assert result is not None
        """
        tmpl = dedent(tmpl).strip()

        tmpl_data = """
            (
                {test_data},    # test data
                r{pattern}   # pattern
            ),
        """
        tmpl_data = dedent(tmpl_data).strip()

        tc_gen = self.tc_gen
        lst = []
        placeholder_table = dict()
        for index, test in enumerate(tc_gen.lst_of_tests):
            _, test_data, _, pattern = test
            test_data = enclose_string(test_data)
            key = '__test_data_placeholder_{}__'.format(index)
            placeholder_table[key] = test_data

            kw = dict(test_data=key,
                      pattern=enclose_string(pattern))
            parametrize_item = tmpl_data.format(**kw)
            parametrize_item = indent(parametrize_item, ' ' * 12)
            lst.append(parametrize_item)

        parametrize_data = '\n'.join(lst).strip()

        for key, value in placeholder_table.items():
            parametrize_data = parametrize_data.replace(key, value)

        test_name = tc_gen.test_name or 'test_generating_script'
        if not test_name.startswith('test_'):
            test_name = 'test_{}'.format(test_name)

        test_script = tmpl.format(
            module_docstring=self.module_docstring,
            test_cls_name=tc_gen.test_cls_name,
            test_name=test_name,
            parametrize_data=parametrize_data
        )
        test_script = '\n'.join(line.rstrip() for line in test_script.splitlines())
        return test_script


class PythonSnippetBuilder:
    """Create Python snippet test script

    Attributes
    ----------
    tc_gen (DynamicGenTestScript): an DynamicGenTestScript instance.
    module_docstring (str): a Python snippet docstring.

    Methods
    -------
    create() -> str
    create_line_via_pattern_script(test_data, pattern) -> str
    create_line_via_patterns_script(test_data, patterns) -> str
    create_multiline_via_pattern_script(test_data, pattern) -> str
    """
    def __init__(self, tc_gen):
        self.tc_gen = tc_gen
        self.module_docstring = create_docstring(
            test_framework='test',
            author=tc_gen.author,
            email=tc_gen.email,
            company=tc_gen.company,
            **tc_gen.kwargs
        )

    def create_line_via_pattern_script(self, test_data, pattern):
        """return python test script - line via pattern

        Parameters
        ----------
        test_data (str): a test data.
        pattern (str): a regex pattern.

        Returns
        -------
        str: a python test script
        """

        tmpl = '''
            {module_docstring}

            import re

            # test data
            test_data = {test_data}

            # regex pattern
            pattern = r{pattern}

            def test_regex(test_data, pattern):
                """test regular expression

                Parameters
                ----------
                test_data (str): a test data
                pattern (str): a regular expression pattern
                """
                print("Pattern: {{}}".format(pattern))
                is_matched = False
                for index, line in enumerate(test_data.splitlines(), 1):
                    match = re.search(pattern, line)
                    if match:
                        is_matched = True
                        if match.groupdict():
                            print("Result{{}}: {{}}".format(index, match.groupdict()))
                        else:
                            print("Result{{}}: {{}}".format(index, match.group()))
                else:
                    if not is_matched:
                        print("Result: No match")

            # function call
            test_regex(test_data, pattern)
        '''
        tmpl = dedent(tmpl).strip()

        test_script = tmpl.format(
            module_docstring=self.module_docstring,
            test_data=test_data,
            pattern=pattern
        )
        return test_script

    def create_line_via_patterns_script(self, test_data, patterns):
        """return python snippet script - line via patterns

        Parameters
        ----------
        test_data (str): a test data.
        patterns (list): a list of patterns.

        Returns
        -------
        str: a python test script
        """
        tmpl = '''
            {module_docstring}
            
            import re
            
            # test data
            test_data = {test_data}
            
            # list of regex pattern
            patterns = [
                {patterns}
            ]
            
            def test_regex(test_data, patterns):
                """test regular expression
            
                Parameters
                ----------
                test_data (str): a test data
                patterns (list): a list of regular expression pattern
                """
            
                for pattern in patterns:
                    is_matched = False
                    for line in test_data.splitlines():
                        match = re.search(pattern, line)
                        if match:
                            is_matched = True
                            print("Pattern: {{}}".format(pattern))
                            if match.groupdict():
                                print("Result: {{}}".format(match.groupdict()))
                            else:
                                print("Result: {{}}".format(match.group()))
                            print("-" * 10)
                    else:
                        if not is_matched:
                            print("Result: No match")
            
            # function call
            test_regex(test_data, patterns)
        '''
        tmpl = dedent(tmpl).strip()

        lst = []
        for pattern in patterns:
            lst.append('r{}'.format(enclose_string(pattern)))
        patterns = indent(',\n'.join(lst), '    ').strip()
        test_script = tmpl.format(
            module_docstring=self.module_docstring,
            test_data=test_data, patterns=patterns
        )
        return test_script

    def create_multiline_via_pattern_script(self, test_data, pattern):
        """return python test script - multiline via pattern

        Parameters
        ----------
        test_data (str): a test data.
        pattern (str): a regex pattern.

        Returns
        -------
        str: a python test script
        """
        tmpl = '''
            {module_docstring}
            
            import re
            
            # test data
            test_data = {test_data}
            
            # regex pattern
            pattern = r{pattern}
            
            def test_regex(test_data, pattern):
                """test regular expression
            
                Parameters
                ----------
                test_data (str): a test data
                pattern (str): a regular expression pattern
                """
            
                print("Pattern: {{}}".format(pattern))
                match = re.search(pattern, test_data)
                if match:
                    if match.groupdict():
                        print("Result: {{}}".format(match.groupdict()))
                    else:
                        print("Result: {{}}".format(match.group()))
                else:
                    print("Result: No match")
            
            # function call
            test_regex(test_data, pattern)
        '''
        tmpl = dedent(tmpl).strip()
        test_script = tmpl.format(
            module_docstring=self.module_docstring,
            test_data=test_data, pattern=pattern
        )
        return test_script

    def create(self):
        """return python test script"""
        tc_gen = self.tc_gen
        if self.tc_gen.is_line:
            if isinstance(tc_gen.test_data, (list, tuple)):
                test_data = enclose_string('\n'.join(tc_gen.test_data))
            else:
                test_data = enclose_string(tc_gen.test_data)

            if len(tc_gen.patterns) == 1:
                pattern = enclose_string(tc_gen.patterns[0])
                test_script = self.create_line_via_pattern_script(
                    test_data, pattern
                )
            else:
                test_script = self.create_line_via_patterns_script(
                    test_data, tc_gen.patterns
                )
        else:
            if isinstance(tc_gen.test_data, (list, tuple)):
                test_data = enclose_string(tc_gen.test_data[0])
            else:
                test_data = enclose_string(tc_gen.test_data)

            pattern = enclose_string(tc_gen.patterns[0])
            test_script = self.create_multiline_via_pattern_script(
                test_data, pattern
            )

        return test_script
