"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const language_1 = require("../../../backend/shared/language");
const transliterator_1 = require("../../../backend/transliterator");
const repository_1 = require("../../../codeartifact/repository");
const monitoring_1 = require("../../../monitoring");
test('basic use', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
    });
    // THEN
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring),
    });
});
test('CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    const codeArtifact = new repository_1.Repository(stack, 'CodeArtifact');
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: codeArtifact.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: codeArtifact.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: codeArtifact.repositoryNpmEndpoint,
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, codeArtifact),
    });
});
test('VPC Endpoints', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, s3 },
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, monitoring, vpc),
    });
});
test('VPC Endpoints and CodeArtifact repository', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'TestStack');
    const bucket = new aws_s3_1.Bucket(stack, 'Bucket');
    const repository = new repository_1.Repository(stack, 'CodeArtifact');
    const monitoring = new monitoring_1.Monitoring(stack, 'Monitoring', {
        alarmActions: { highSeverity: 'high-sev', normalSeverity: 'normal-sev' },
    });
    const vpc = new aws_ec2_1.Vpc(stack, 'VPC', { subnetConfiguration: [{ name: 'Isolated', subnetType: aws_ec2_1.SubnetType.ISOLATED }] });
    const codeArtifactApi = vpc.addInterfaceEndpoint('CodeArtifact.API', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.api'),
    });
    const codeArtifact = vpc.addInterfaceEndpoint('CodeArtifact.Repo', {
        service: new aws_ec2_1.InterfaceVpcEndpointAwsService('codeartifact.repositories'),
    });
    const s3 = vpc.addGatewayEndpoint('S3', {
        service: aws_ec2_1.GatewayVpcEndpointAwsService.S3,
    });
    // WHEN
    new transliterator_1.Transliterator(stack, 'Transliterator', {
        bucket,
        codeArtifact: repository,
        language: language_1.DocumentationLanguage.PYTHON,
        monitoring,
        vpc,
        vpcEndpoints: { codeArtifactApi, codeArtifact, s3 },
        vpcSubnets: { subnetType: aws_ec2_1.SubnetType.ISOLATED },
    });
    // THEN
    expect(stack).toHaveResourceLike('AWS::Lambda::Function', {
        Environment: {
            Variables: stack.resolve({
                CODE_ARTIFACT_DOMAIN_NAME: repository.repositoryDomainName,
                CODE_ARTIFACT_DOMAIN_OWNER: repository.repositoryDomainOwner,
                CODE_ARTIFACT_REPOSITORY_ENDPOINT: repository.repositoryNpmEndpoint,
                CODE_ARTIFACT_API_ENDPOINT: core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, codeArtifactApi.vpcEndpointDnsEntries))),
            }),
        },
    });
    expect(app.synth().getStackByName(stack.stackName).template).toMatchSnapshot({
        Outputs: expect.anything(),
        Parameters: expect.anything(),
        Resources: ignoreResources(stack, bucket, repository, monitoring, vpc),
    });
});
/**
 * Creates a property matcher that expects "anything" for all CfnResource
 * logicalIds found within the provided constructs.
 */
function ignoreResources(stack, ...constructs) {
    const result = {};
    for (const resource of resourcesIn(...constructs)) {
        result[stack.resolve(resource.logicalId)] = expect.anything();
    }
    return result;
    function* resourcesIn(...scopes) {
        for (const scope of scopes) {
            if (core_1.CfnResource.isCfnResource(scope)) {
                yield scope;
            }
            yield* resourcesIn(...scope.node.children);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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