"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const crypto_1 = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const language_1 = require("./backend/shared/language");
const deep_link_1 = require("./deep-link");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new aws_cloudwatch_1.Dashboard(this, 'Resource', {
            dashboardName: props.dashboardName,
            periodOverride: aws_cloudwatch_1.PeriodOverride.INHERIT,
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Catalog Overview',
                            '',
                            `[button:Package Data](${deep_link_1.s3ObjectUrl(props.packageData)})`,
                            `[button:Catalog Builder](${deep_link_1.lambdaFunctionUrl(props.orchestration.catalogBuilder)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props.inventory),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Discovery/Follower Function',
                            '',
                            `[button:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.discovery.follow)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            fillMetric(props.discovery.follow.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.discovery.follow.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.discovery.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Metrics',
                        left: [
                            props.discovery.metricChangeCount({ label: 'Change Count' }),
                            props.discovery.metricUnprocessableEntity({ label: 'Unprocessable' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            fillMetric(props.discovery.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            fillMetric(props.discovery.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(15),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Discovery/Stage Function',
                            '',
                            `[button:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.discovery.follow)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            fillMetric(props.discovery.stage.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.discovery.stage.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.discovery.metricStagingTime({ label: 'Staging Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.discovery.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.discovery.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.discovery.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.discovery.stage.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.discovery.stage.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.discovery.stage.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:State Machine](${deep_link_1.stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${deep_link_1.lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Reprocess](${deep_link_1.lambdaFunctionUrl(props.orchestration.reprocessAllFunction)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                // deny list
                // ----------------------------------------------
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Deny List',
                            '',
                            `[button:Deny List Object](${deep_link_1.s3ObjectUrl(props.denyList.bucket, props.denyList.objectKey)})`,
                            `[button:Prune Function](${deep_link_1.lambdaFunctionUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Prune Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Delete Queue](${deep_link_1.sqsQueueUrl(props.denyList.prune.queue)})`,
                            `[button:Delete Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.deleteHandler)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Deny List',
                        left: [
                            fillMetric(props.denyList.metricDenyListRules({ label: 'Rules' }), 'REPEAT'),
                            props.denyList.prune.queue.metricNumberOfMessagesDeleted({ label: 'Deleted Files' }),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Prune Function Health',
                        left: [
                            fillMetric(props.denyList.prune.pruneHandler.metricInvocations({ label: 'Invocations' })),
                            fillMetric(props.denyList.prune.pruneHandler.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ],
            ],
        });
    }
    *catalogOverviewLanguageSections(inventory) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: '# Documentation Generation',
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function fillMetric(metric, value = 0) {
    // We assume namespace + name is enough to uniquely identify a metric here.
    // This is true locally at this time, but in case this ever changes, consider
    // also processing dimensions and period.
    const h = crypto_1.createHash('sha256')
        .update(metric.namespace)
        .update('\0')
        .update(metric.metricName)
        .digest('hex');
    const metricName = `m${h}`;
    return new aws_cloudwatch_1.MathExpression({
        expression: `FILL(${metricName}, ${value})`,
        label: metric.label,
        usingMetrics: { [metricName]: metric },
    });
}
//# sourceMappingURL=data:application/json;base64,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