"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const Nano = require("nano");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const constants_1 = require("./constants");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const TIMEOUT_MILLISECONDS = 10000;
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const marker = await loadLastTransactionMarker(4401405 /* around December 30, 2019 */);
    const config = {
        includeDocs: true,
        // pause the changes reader after each request
        wait: true,
        since: marker.lastSeq.toFixed(),
        // `changesReader.get` stops once a response with zero changes is received, however it waits too long
        //  since we want to terminate the Lambda function we define a timeout shorter than the default
        timeout: TIMEOUT_MILLISECONDS,
        // Only items with a name
        selector: {
            name: { $gt: null },
        },
        batchSize: 30,
    };
    const nano = Nano(NPM_REPLICA_REGISTRY_URL);
    const db = nano.db.use('registry');
    // We need to make an explicit Promise here, because otherwise Lambda won't
    // know when it's done...
    return new Promise((ok, ko) => {
        let updatedMarker = marker;
        db.changesReader.get(config)
            .on('batch', aws_embedded_metrics_1.metricScope((metrics) => async (batch) => {
            var _a;
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            metrics.setProperty('StartSeq', updatedMarker);
            const startTime = Date.now();
            // Emit npm.js replication lag
            for (const { doc } of batch) {
                if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                    metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - new Date(doc.time.modified).getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                }
            }
            try {
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                const lastSeq = Math.max(...batch.map((change) => change.seq));
                metrics.setProperty('EndSeq', updatedMarker);
                // Obtain the modified package version from the update event, and filter
                // out packages that are not of interest to us (not construct libraries).
                const versionInfos = getRelevantVersionInfos(batch, metrics, denyList);
                console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                // since npm will include ALL versions in every single change record
                // we filter those we have already seen to reduce the throttling probability caused by
                // reprocessing the same version over and over again.
                const newVersions = versionInfos.filter(({ infos, modified }) => {
                    const key = `${infos.name}@${infos.version}`;
                    if (marker.knownPackageVersions.has(key) && marker.knownPackageVersions.get(key) >= modified) {
                        // We already saw this package update, or a more recent one.
                        console.log(`Ignoring change for ${key} since we've already seen it before`);
                        return false;
                    }
                    marker.knownPackageVersions.set(key, modified);
                    return true;
                });
                // Notify the staging & notification function
                console.log(`Sending ${newVersions.length} packages for staging`);
                await aws.sqsSendMessageBatch(queueUrl, newVersions);
                // Update the transaction marker in S3.
                await saveLastTransactionMarker(lastSeq);
                // If we have enough time left before timeout, proceed with the next batch, otherwise we're done here.
                // Since the distribution of the time it takes to process each package/batch is non uniform, this is a best
                // effort, and we expect the function to timeout in some invocations, we rely on the downstream idempotency to handle this.
                if (context.getRemainingTimeInMillis() >= 30000 /* 30 seconds */) {
                    console.log('There is still time, requesting the next batch...');
                    // Note: the `resume` function is missing from the `nano` type definitions, but is there...
                    db.changesReader.resume();
                }
                else {
                    console.log('We are almost out of time, so stopping here.');
                    db.changesReader.stop();
                    metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
                    ok();
                }
            }
            catch (err) {
                // An exception bubbled out, which means this Lambda execution has failed.
                console.error(`Unexpected error: ${err}`);
                db.changesReader.stop();
                ko(err);
            }
            finally {
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
            }
        }))
            .once('end', () => {
            console.log('No more updates to process, exiting.');
            ok();
        });
    });
    //#region Last transaction marker
    /**
     * Loads the last transaction marker from S3.
     *
     * @param defaultValue the value to return in case the marker does not exist
     *
     * @returns the value of the last transaction marker.
     */
    async function loadLastTransactionMarker(defaultValue) {
        try {
            const response = await aws.s3().getObject({
                Bucket: stagingBucket,
                Key: constants_1.DISCOVERY_MARKER_KEY,
            }).promise();
            let result = JSON.parse(response.Body.toString('utf-8'), (key, value) => {
                if (key !== 'knownPackageVersions' || typeof value !== 'object') {
                    return value;
                }
                return Object.entries(value).reduce((map, [pkg, time]) => map.set(pkg, new Date(time)), new Map());
            });
            if (typeof result === 'number') {
                console.log('Migrating transaction marker to new format...');
                result = { lastSeq: result, knownPackageVersions: new Map() };
            }
            console.log(`Read last transaction marker: ${result.lastSeq}`);
            return result;
        }
        catch (error) {
            if (error.code !== 'NoSuchKey') {
                throw error;
            }
            console.log(`Marker object (s3://${stagingBucket}/${constants_1.DISCOVERY_MARKER_KEY}) does not exist, starting from the default (${defaultValue})`);
            return { lastSeq: defaultValue, knownPackageVersions: new Map() };
        }
    }
    /**
     * Updates the last transaction marker in S3.
     *
     * @param sequence the last transaction marker value
     */
    async function saveLastTransactionMarker(sequence) {
        console.log(`Updating last transaction marker to ${sequence}`);
        return aws.s3PutObject(context, stagingBucket, constants_1.DISCOVERY_MARKER_KEY, sequence.toFixed(), { ContentType: 'text/plain; charset=UTF-8' });
    }
}
exports.handler = handler;
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList) {
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const sortedUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)])
            // Sort by date, descending
            .sort(([, l], [, r]) => r.getTime() - l.getTime());
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, sortedUpdates.length, aws_embedded_metrics_1.Unit.Count);
        for (const [version, modified] of sortedUpdates) {
            const infos = change.doc.versions[version];
            if (infos == null) {
                // Could be the version in question was un-published.
                console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
            }
            else if (isConstructLibrary(infos)) {
                // skip if this package is denied
                const denied = denyList.lookup(infos.name, infos.version);
                if (denied) {
                    console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                    metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                    continue;
                }
                metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                result.push({ infos, modified, seq: change.seq });
            }
            else {
                console.log(`[${change.seq}] Ignoring "${change.doc.name}@${version}" as it is not a construct library.`);
            }
        }
    }
    return result;
    function isConstructLibrary(infos) {
        var _a;
        if (infos.jsii == null) {
            return false;
        }
        return infos.name === 'construct'
            || infos.name === 'aws-cdk-lib'
            || infos.name.startsWith('@aws-cdk')
            || ((_a = infos.keywords) === null || _a === void 0 ? void 0 : _a.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
}
//# sourceMappingURL=data:application/json;base64,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