"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const constants_1 = require("./constants");
const follow_1 = require("./follow");
const stage_1 = require("./stage");
/**
 * This discovery function periodically scans the CouchDB replica of npmjs.com
 * to discover newly published packages that are relevant for indexing in the
 * Construct Hub, then notifies the ingestion function about those.
 */
class Discovery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.timeout = core_1.Duration.minutes(15);
        this.bucket = new aws_s3_1.Bucket(this, 'StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [
                {
                    prefix: "staged/" /* STAGED_KEY_PREFIX */,
                    expiration: core_1.Duration.days(30),
                },
            ],
        });
        this.queue = new aws_sqs_1.Queue(this, 'DiscoveredPackages', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            // This is a Lambda event source, visibility timeout needs to be >= to target function timeout
            visibilityTimeout: this.timeout,
        });
        this.follow = new follow_1.Follow(this, 'Default', {
            description: '[ConstructHub/Discovery/NpmCatalogFollower] Periodically query npm.js index for new construct libraries',
            memorySize: 10240,
            /// Only one execution (avoids race conditions on the S3 marker object)
            reservedConcurrentExecutions: 1,
            timeout: this.timeout,
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: this.queue.queueUrl,
            },
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.queue.grantSendMessages(this.follow);
        this.bucket.grantReadWrite(this.follow, constants_1.DISCOVERY_MARKER_KEY);
        props.denyList.grantRead(this.follow);
        this.stage = new stage_1.Stage(this, 'Stage', {
            deadLetterQueueEnabled: true,
            description: '[Discovery/StageAndNotify] Stages a new package version and notifies Construct Hub about it',
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: props.queue.queueUrl,
            },
        });
        this.bucket.grantReadWrite(this.stage, `${"staged/" /* STAGED_KEY_PREFIX */}*`);
        props.queue.grantSendMessages(this.stage);
        this.stage.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue));
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(this.timeout),
            targets: [new aws_events_targets_1.LambdaFunction(this.follow)],
        });
        this.alarmErrors = this.follow.metricErrors({ period: core_1.Duration.minutes(15) })
            .createAlarm(this, 'ErrorsAlarm', {
            alarmName: `${this.node.path}/Errors`,
            alarmDescription: [
                'The discovery/follow function (on npmjs.com) failed to run',
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.follow)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        this.alarmNoInvocations = this.follow.metricInvocations({ period: core_1.Duration.minutes(15) })
            .createAlarm(this, 'NoInvocationsAlarm', {
            alarmName: `${this.node.path}/NotRunning`,
            alarmDescription: [
                'The discovery/follow function (on npmjs.com) is not running as scheduled',
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.follow)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
        this.bucket.grantReadWrite(this.stage);
        props.queue.grantSendMessages(this.stage);
        props.monitoring.addHighSeverityAlarm('Discovery Failures', this.alarmErrors);
        props.monitoring.addHighSeverityAlarm('Discovery not Running', this.alarmNoInvocations);
        this.alarmDeadLetterQueueNotEmpty = this.stage.deadLetterQueue.metricApproximateNumberOfMessagesVisible()
            .createAlarm(this, 'AlarmDLQ', {
            alarmDescription: 'The dead-letter-queue associated with the discovery stage-and-notify function is not empty',
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        });
    }
    /**
     * The average time it took to process a changes batch.
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of changes that were processed.
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The age of the oldest package version that was processed.
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were inspected.
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were deemed relevant.
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of time that was remaining when the lambda returned in order to
     * avoid hitting a timeout.
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The average time it took to stage a package to S3.
     */
    metricStagingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "StagingTime" /* STAGING_TIME */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of changes that were not processed due to having an invalid
     * format.
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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