"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const https = require("https");
const url_1 = require("url");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const aws = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const constants_1 = require("./constants");
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    metrics.setNamespace(constants_1.METRICS_NAMESPACE);
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const messages = await Promise.all(event.Records
        .map((record) => JSON.parse(record.body, (k, v) => k === 'modified' ? new Date(v) : v))
        .map(stageUpdatedVersion));
    console.log(`Sending ${messages.length} for ingestion`);
    await aws.sqsSendMessageBatch(queueUrl, messages);
    async function stageUpdatedVersion({ infos, modified, seq }) {
        const startTime = Date.now();
        const tarball = await httpGet(infos.dist.tarball);
        // Store the tarball into the staging bucket
        // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
        // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
        const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(infos.dist.tarball).pathname}`.replace(/\/{2,}/g, '/');
        await aws.s3PutObject(context, bucket, stagingKey, tarball, {
            Metadata: {
                'Modified-At': modified.toISOString(),
                'Origin-Integrity': infos.dist.shasum,
                'Origin-Uri': infos.dist.tarball,
                'Sequence': seq.toFixed(),
            },
        });
        metrics.putMetric("StagingTime" /* STAGING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
        console.log(`Uploaded: ${stagingKey}`);
        // Prepare SQS message for ingestion
        const messageBase = {
            tarballUri: `s3://${bucket}/${stagingKey}`,
            metadata: {
                dist: infos.dist.tarball,
                seq: seq.toFixed(),
            },
            time: modified.toUTCString(),
        };
        return {
            ...messageBase,
            integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
        };
    }
});
/**
 * Makes an HTTP GET request, and returns the resulting payload.
 *
 * @param url the URL to get.
 *
 * @returns a Buffer containing the received data.
 */
function httpGet(url) {
    return new Promise((ok, ko) => {
        https.get(url, (response) => {
            if (response.statusCode !== 200) {
                throw new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`);
            }
            let body = Buffer.alloc(0);
            response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
            response.once('close', () => ok(body));
            response.once('error', ko);
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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