"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inventory = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const canary_1 = require("./canary");
const constants_1 = require("./constants");
/**
 * Periodically computes an inventory of all indexed packages into the storage
 * bucket, and produces metrics with an overview of the index' state.
 */
class Inventory extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const rate = (_a = props.scheduleRate) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(5);
        this.canary = new canary_1.Canary(this, 'Resource', {
            description: '[ConstructHub/Inventory] A canary that periodically inspects the list of indexed packages',
            environment: { BUCKET_NAME: props.bucket.bucketName },
            logRetention: props.logRetention,
            memorySize: 10240,
            timeout: rate,
        });
        const grant = props.bucket.grantRead(this.canary);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(rate),
            targets: [new aws_events_targets_1.LambdaFunction(this.canary)],
        }).node.addDependency(grant);
        props.monitoring.addHighSeverityAlarm('Inventory Canary is not Running', this.canary.metricInvocations({ period: core_1.Duration.minutes(5) }).createAlarm(this, 'Not Running', {
            alarmName: `${this.node.path}/NotRunning`,
            alarmDescription: [
                'The inventory canary is not running!',
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        props.monitoring.addHighSeverityAlarm('Inventory Canary is failing', this.canary.metricErrors({ period: core_1.Duration.minutes(5) }).createAlarm(this, 'Failures', {
            alarmName: `${this.node.path}/Failures`,
            alarmDescription: [
                'The inventory canary is failing!',
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
    }
    metricMissingPackageMetadataCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingAssemblyCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageCount" /* PACKAGE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageMajorCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingPackageTarballCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricSubmoduleCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "SubmoduleCount" /* SUBMODULE_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricUnknownObjectCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages for which all versions are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions for which all versions are missing a
     * documnetation artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that are missing a documnetation
     * artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that have at least one version for which there is
     * available documentation in the provided `DocumentationLanguage`.
     */
    metricSupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that have at least one version for
     * which there is available documentation in the provided
     * `DocumentationLanguage`.
     */
    metricSupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that have available documentation in the
     * provided `DocumentationLanguage`.
     */
    metricSupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that have available documentation
     * in the provided `DocumentationLanguage`.
     */
    metricSupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_1.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.Inventory = Inventory;
//# sourceMappingURL=data:application/json;base64,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