"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const catalog_builder_1 = require("../catalog-builder");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const reprocess_all_1 = require("./reprocess-all");
const SUPPORTED_LANGUAGES = [language_1.DocumentationLanguage.PYTHON, language_1.DocumentationLanguage.TYPESCRIPT];
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props).function;
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 })
            .addCatch(new aws_stepfunctions_1.Pass(this, 'Add to catalog.json failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, 'Add to catalog.json fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => new tasks.LambdaInvoke(this, `Generate ${language} docs`, {
            lambdaFunction: new transliterator_1.Transliterator(this, `DocGen-${language}`, { ...props, language }).function,
            outputPath: '$.result',
            resultSelector: {
                result: {
                    'language': language,
                    'success.$': '$.Payload',
                },
            },
        }).addRetry({ errors: ['Lambda.TooManyRequestsException'], interval: core_1.Duration.seconds(30), maxAttempts: 5 })
            .addCatch(new aws_stepfunctions_1.Pass(this, `Generate ${language} docs failure`, { parameters: { 'error.$': 'States.StringToJson($.Cause)', language } }), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, `Generate ${language} docs fault`, { parameters: { 'error.$': '$.Cause', language } }), { errors: ['States.ALL'] })))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineType: aws_stepfunctions_1.StateMachineType.STANDARD,
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // This function is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the back-end.
        this.reprocessAllFunction = new reprocess_all_1.ReprocessAll(this, 'ReprocessAll', {
            description: '[ConstructHub/ReprocessAll] Reprocess all package versions through the backend',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.bucket.grantRead(this.reprocessAllFunction);
        this.stateMachine.grantStartExecution(this.reprocessAllFunction);
    }
}
exports.Orchestration = Orchestration;
//# sourceMappingURL=data:application/json;base64,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