"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.reset = exports.sqsSendMessageBatch = exports.lambda = exports.stepFunctions = exports.sqs = exports.s3ObjectExists = exports.s3PutObject = exports.s3 = void 0;
const _AWS = require("aws-sdk");
const AWSXRay = require("aws-xray-sdk-core");
// Do nothing if there is no XRay trace context
AWSXRay.setContextMissingStrategy(() => { });
const AWS = AWSXRay.captureAWS(_AWS);
let _s3;
let _sqs;
let _sfn;
let _lambda;
;
function s3() {
    if (_s3 == null) {
        _s3 = new AWS.S3();
    }
    return _s3;
}
exports.s3 = s3;
/**
 * Puts an object in an S3 bucket while adding metadata corresponding to the
 * current Lambda context.
 *
 * @param context the Lambda context that is being used.
 * @param bucket  the S3 bucket where to put the object.
 * @param key     the S3 key to be used.
 * @param body    the body of the object to be put.
 * @param opts    additional options for the S3 request.
 */
function s3PutObject(context, bucket, key, body, opts = {}) {
    return s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            ...opts.Metadata,
        },
        ...opts,
    }).promise();
}
exports.s3PutObject = s3PutObject;
/**
 * Checks whether an object exists in S3 at the provided bucket and key.
 */
function s3ObjectExists(bucket, key) {
    return s3().headObject({
        Bucket: bucket,
        Key: key,
    }).promise()
        .then(() => true, (cause) => {
        if (cause.code === 'NotFound') {
            return false;
        }
        return Promise.reject(cause);
    });
}
exports.s3ObjectExists = s3ObjectExists;
function sqs() {
    if (_sqs == null) {
        _sqs = new AWS.SQS();
    }
    return _sqs;
}
exports.sqs = sqs;
function stepFunctions() {
    if (_sfn == null) {
        _sfn = new AWS.StepFunctions();
    }
    return _sfn;
}
exports.stepFunctions = stepFunctions;
function lambda() {
    if (_lambda == null) {
        _lambda = new AWS.Lambda();
    }
    return _lambda;
}
exports.lambda = lambda;
/**
 * Sends the supplied messages to the designated SQS queue url using the
 * `SendMessageBatch` API, for maximum throughput. This encodes the provided
 * messages using `JSON.stringify` and creates batches as large as possible.
 *
 * @param queueUrl the URL of the queue where to send the messages
 * @param messages the messages to be sent
 */
async function sqsSendMessageBatch(queueUrl, messages) {
    var _a;
    for (const batch of batchedMessages()) {
        const result = await sqs().sendMessageBatch({ Entries: batch, QueueUrl: queueUrl }).promise();
        if (result.Failed.length > 0) {
            for (const { Id, SenderFault, Code, Message } of result.Failed) {
                const faultType = SenderFault ? 'sender fault' : 'server fault';
                const payload = (_a = batch.find((msg) => msg.Id === Id)) === null || _a === void 0 ? void 0 : _a.MessageBody;
                console.error(`Failed sending message ${Id} due to a ${faultType} (${Code} - ${Message}): ${payload}`);
            }
            throw new Error(`Failed to send ${result.Failed.length} messages!`);
        }
    }
    /**
     * The SQS.SendMessageBatch call allows to batch up messages for up to 256KiB
     * (262_144 bytes) worth of message data, or up to 10 messages (whicever
     * happens first). This generator batches the provided messages in such a way
     * that this limit is consistently maxed out.
     *
     * @param messages the messages to be batched. They will be stringified using
     *                 `JSON.stringify` with no indentation setting.
     */
    function* batchedMessages() {
        const MAX_PAYLOAD_SIZE = 262144;
        let batch = new Array();
        let totalPayloadSize = 0;
        for (const message of messages) {
            const payload = JSON.stringify(message);
            const payloadSize = Buffer.from(payload, 'utf-8').length;
            if (totalPayloadSize + payloadSize > MAX_PAYLOAD_SIZE || batch.length === 10) {
                yield batch;
                batch = [];
                totalPayloadSize = 0;
            }
            batch.push({ Id: batch.length.toFixed(), MessageBody: payload });
            totalPayloadSize += payloadSize;
        }
        if (batch.length > 0) {
            yield batch;
        }
    }
}
exports.sqsSendMessageBatch = sqsSendMessageBatch;
/**
 * Resets all clients vended by this module. This is useful in unit tests when
 * `aws-sdk-mocks` is used, so that new mocks are injected as intended.
 */
function reset() {
    _s3 = _sqs = _sfn = _lambda = undefined;
}
exports.reset = reset;
//# sourceMappingURL=data:application/json;base64,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