"use strict";
const assert_1 = require("@aws-cdk/assert");
const cdk = require("@aws-cdk/core");
const codepipeline = require("../lib");
const stage_1 = require("../lib/stage");
module.exports = {
    'Pipeline Stages': {
        'can be inserted before another Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const secondStage = pipeline.addStage({ stageName: 'SecondStage' });
            pipeline.addStage({
                stageName: 'FirstStage',
                placement: {
                    rightBefore: secondStage,
                },
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    { 'Name': 'FirstStage' },
                    { 'Name': 'SecondStage' },
                ],
            }));
            test.done();
        },
        'can be inserted after another Stage'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const firstStage = pipeline.addStage({ stageName: 'FirstStage' });
            pipeline.addStage({ stageName: 'ThirdStage' });
            pipeline.addStage({
                stageName: 'SecondStage',
                placement: {
                    justAfter: firstStage,
                },
            });
            assert_1.expect(stack, true).to(assert_1.haveResourceLike('AWS::CodePipeline::Pipeline', {
                'Stages': [
                    { 'Name': 'FirstStage' },
                    { 'Name': 'SecondStage' },
                    { 'Name': 'ThirdStage' },
                ],
            }));
            test.done();
        },
        "attempting to insert a Stage before a Stage that doesn't exist results in an error"(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            const anotherPipeline = new codepipeline.Pipeline(stack, 'AnotherPipeline');
            test.throws(() => {
                anotherPipeline.addStage({
                    stageName: 'AnotherStage',
                    placement: {
                        rightBefore: stage,
                    },
                });
            }, /before/i);
            test.done();
        },
        "attempting to insert a Stage after a Stage that doesn't exist results in an error"(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            const anotherPipeline = new codepipeline.Pipeline(stack, 'AnotherPipeline');
            test.throws(() => {
                anotherPipeline.addStage({
                    stageName: 'AnotherStage',
                    placement: {
                        justAfter: stage,
                    },
                });
            }, /after/i);
            test.done();
        },
        'providing more than one placement value results in an error'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline');
            const stage = pipeline.addStage({ stageName: 'Stage' });
            test.throws(() => {
                pipeline.addStage({
                    stageName: 'SecondStage',
                    placement: {
                        rightBefore: stage,
                        justAfter: stage,
                    },
                });
                // incredibly, an arrow function below causes nodeunit to crap out with:
                // "TypeError: Function has non-object prototype 'undefined' in instanceof check"
                // tslint:disable-next-line:only-arrow-functions
            }, function (e) {
                return /rightBefore/.test(e) && /justAfter/.test(e);
            });
            test.done();
        },
        'can be retrieved from a pipeline after it has been created'(test) {
            const stack = new cdk.Stack();
            const pipeline = new codepipeline.Pipeline(stack, 'Pipeline', {
                stages: [
                    {
                        stageName: 'FirstStage',
                    },
                ],
            });
            pipeline.addStage({ stageName: 'SecondStage' });
            test.equal(pipeline.stages.length, 2);
            test.equal(pipeline.stages[0].stageName, 'FirstStage');
            test.equal(pipeline.stages[1].stageName, 'SecondStage');
            // adding stages to the returned array should have no effect
            pipeline.stages.push(new stage_1.Stage({
                stageName: 'ThirdStage',
            }, pipeline));
            test.equal(pipeline.stageCount, 2);
            test.done();
        },
    },
};
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zdGFnZXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnN0YWdlcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEsNENBQTJEO0FBQzNELHFDQUFxQztBQUVyQyx1Q0FBdUM7QUFDdkMsd0NBQXFDO0FBSXJDLGlCQUFTO0lBQ1AsaUJBQWlCLEVBQUU7UUFDakIsc0NBQXNDLENBQUMsSUFBVTtZQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBRTlELE1BQU0sV0FBVyxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRSxTQUFTLEVBQUUsYUFBYSxFQUFFLENBQUMsQ0FBQztZQUNwRSxRQUFRLENBQUMsUUFBUSxDQUFDO2dCQUNoQixTQUFTLEVBQUUsWUFBWTtnQkFDdkIsU0FBUyxFQUFFO29CQUNULFdBQVcsRUFBRSxXQUFXO2lCQUN6QjthQUNGLENBQUMsQ0FBQztZQUVILGVBQU0sQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUMsRUFBRSxDQUFDLHlCQUFnQixDQUFDLDZCQUE2QixFQUFFO2dCQUNyRSxRQUFRLEVBQUU7b0JBQ1IsRUFBRSxNQUFNLEVBQUUsWUFBWSxFQUFFO29CQUN4QixFQUFFLE1BQU0sRUFBRSxhQUFhLEVBQUU7aUJBQzFCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQscUNBQXFDLENBQUMsSUFBVTtZQUM5QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBRTlELE1BQU0sVUFBVSxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRSxTQUFTLEVBQUUsWUFBWSxFQUFFLENBQUMsQ0FBQztZQUNsRSxRQUFRLENBQUMsUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLFlBQVksRUFBRSxDQUFDLENBQUM7WUFDL0MsUUFBUSxDQUFDLFFBQVEsQ0FBQztnQkFDaEIsU0FBUyxFQUFFLGFBQWE7Z0JBQ3hCLFNBQVMsRUFBRTtvQkFDVCxTQUFTLEVBQUUsVUFBVTtpQkFDdEI7YUFDRixDQUFDLENBQUM7WUFFSCxlQUFNLENBQUMsS0FBSyxFQUFFLElBQUksQ0FBQyxDQUFDLEVBQUUsQ0FBQyx5QkFBZ0IsQ0FBQyw2QkFBNkIsRUFBRTtnQkFDckUsUUFBUSxFQUFFO29CQUNSLEVBQUUsTUFBTSxFQUFFLFlBQVksRUFBRTtvQkFDeEIsRUFBRSxNQUFNLEVBQUUsYUFBYSxFQUFFO29CQUN6QixFQUFFLE1BQU0sRUFBRSxZQUFZLEVBQUU7aUJBQ3pCO2FBQ0YsQ0FBQyxDQUFDLENBQUM7WUFFSixJQUFJLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDZCxDQUFDO1FBRUQsb0ZBQW9GLENBQUMsSUFBVTtZQUM3RixNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztZQUM5QixNQUFNLFFBQVEsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1lBQzlELE1BQU0sS0FBSyxHQUFHLFFBQVEsQ0FBQyxRQUFRLENBQUMsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLENBQUMsQ0FBQztZQUV4RCxNQUFNLGVBQWUsR0FBRyxJQUFJLFlBQVksQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLGlCQUFpQixDQUFDLENBQUM7WUFDNUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxHQUFHLEVBQUU7Z0JBQ2YsZUFBZSxDQUFDLFFBQVEsQ0FBQztvQkFDdkIsU0FBUyxFQUFFLGNBQWM7b0JBQ3pCLFNBQVMsRUFBRTt3QkFDVCxXQUFXLEVBQUUsS0FBSztxQkFDbkI7aUJBQ0YsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBRWQsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQ2QsQ0FBQztRQUVELG1GQUFtRixDQUFDLElBQVU7WUFDNUYsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7WUFDOUIsTUFBTSxRQUFRLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxVQUFVLENBQUMsQ0FBQztZQUM5RCxNQUFNLEtBQUssR0FBRyxRQUFRLENBQUMsUUFBUSxDQUFDLEVBQUUsU0FBUyxFQUFFLE9BQU8sRUFBRSxDQUFDLENBQUM7WUFFeEQsTUFBTSxlQUFlLEdBQUcsSUFBSSxZQUFZLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxpQkFBaUIsQ0FBQyxDQUFDO1lBQzVFLElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLGVBQWUsQ0FBQyxRQUFRLENBQUM7b0JBQ3ZCLFNBQVMsRUFBRSxjQUFjO29CQUN6QixTQUFTLEVBQUU7d0JBQ1QsU0FBUyxFQUFFLEtBQUs7cUJBQ2pCO2lCQUNGLENBQUMsQ0FBQztZQUNMLENBQUMsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUViLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw2REFBNkQsQ0FBQyxJQUFVO1lBQ3RFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxDQUFDLENBQUM7WUFDOUQsTUFBTSxLQUFLLEdBQUcsUUFBUSxDQUFDLFFBQVEsQ0FBQyxFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO1lBRXhELElBQUksQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO2dCQUNmLFFBQVEsQ0FBQyxRQUFRLENBQUM7b0JBQ2hCLFNBQVMsRUFBRSxhQUFhO29CQUN4QixTQUFTLEVBQUU7d0JBQ1QsV0FBVyxFQUFFLEtBQUs7d0JBQ2xCLFNBQVMsRUFBRSxLQUFLO3FCQUNqQjtpQkFDRixDQUFDLENBQUM7Z0JBQ0wsd0VBQXdFO2dCQUN4RSxpRkFBaUY7Z0JBQ2pGLGdEQUFnRDtZQUNoRCxDQUFDLEVBQUUsVUFBUyxDQUFNO2dCQUNoQixPQUFPLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLElBQUksV0FBVyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsQ0FBQztZQUN0RCxDQUFDLENBQUMsQ0FBQztZQUVILElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7UUFFRCw0REFBNEQsQ0FBQyxJQUFVO1lBQ3JFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO1lBQzlCLE1BQU0sUUFBUSxHQUFHLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO2dCQUM1RCxNQUFNLEVBQUU7b0JBQ047d0JBQ0UsU0FBUyxFQUFFLFlBQVk7cUJBQ3hCO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDO1lBRUgsUUFBUSxDQUFDLFFBQVEsQ0FBQyxFQUFFLFNBQVMsRUFBRSxhQUFhLEVBQUUsQ0FBQyxDQUFDO1lBRWhELElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDLENBQUM7WUFDdEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsRUFBRSxZQUFZLENBQUMsQ0FBQztZQUN2RCxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUyxFQUFFLGFBQWEsQ0FBQyxDQUFDO1lBRXhELDREQUE0RDtZQUM1RCxRQUFRLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLGFBQUssQ0FBQztnQkFDN0IsU0FBUyxFQUFFLFlBQVk7YUFDeEIsRUFBRSxRQUFRLENBQUMsQ0FBQyxDQUFDO1lBQ2QsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO1lBRW5DLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNkLENBQUM7S0FDRjtDQUNGLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBleHBlY3QsIGhhdmVSZXNvdXJjZUxpa2UgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgVGVzdCB9IGZyb20gJ25vZGV1bml0JztcbmltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tICcuLi9saWInO1xuaW1wb3J0IHsgU3RhZ2UgfSBmcm9tICcuLi9saWIvc3RhZ2UnO1xuXG4vLyB0c2xpbnQ6ZGlzYWJsZTpvYmplY3QtbGl0ZXJhbC1rZXktcXVvdGVzXG5cbmV4cG9ydCA9IHtcbiAgJ1BpcGVsaW5lIFN0YWdlcyc6IHtcbiAgICAnY2FuIGJlIGluc2VydGVkIGJlZm9yZSBhbm90aGVyIFN0YWdlJyh0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHBpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShzdGFjaywgJ1BpcGVsaW5lJyk7XG5cbiAgICAgIGNvbnN0IHNlY29uZFN0YWdlID0gcGlwZWxpbmUuYWRkU3RhZ2UoeyBzdGFnZU5hbWU6ICdTZWNvbmRTdGFnZScgfSk7XG4gICAgICBwaXBlbGluZS5hZGRTdGFnZSh7XG4gICAgICAgIHN0YWdlTmFtZTogJ0ZpcnN0U3RhZ2UnLFxuICAgICAgICBwbGFjZW1lbnQ6IHtcbiAgICAgICAgICByaWdodEJlZm9yZTogc2Vjb25kU3RhZ2UsXG4gICAgICAgIH0sXG4gICAgICB9KTtcblxuICAgICAgZXhwZWN0KHN0YWNrLCB0cnVlKS50byhoYXZlUmVzb3VyY2VMaWtlKCdBV1M6OkNvZGVQaXBlbGluZTo6UGlwZWxpbmUnLCB7XG4gICAgICAgICdTdGFnZXMnOiBbXG4gICAgICAgICAgeyAnTmFtZSc6ICdGaXJzdFN0YWdlJyB9LFxuICAgICAgICAgIHsgJ05hbWUnOiAnU2Vjb25kU3RhZ2UnIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICAnY2FuIGJlIGluc2VydGVkIGFmdGVyIGFub3RoZXIgU3RhZ2UnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgcGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHN0YWNrLCAnUGlwZWxpbmUnKTtcblxuICAgICAgY29uc3QgZmlyc3RTdGFnZSA9IHBpcGVsaW5lLmFkZFN0YWdlKHsgc3RhZ2VOYW1lOiAnRmlyc3RTdGFnZScgfSk7XG4gICAgICBwaXBlbGluZS5hZGRTdGFnZSh7IHN0YWdlTmFtZTogJ1RoaXJkU3RhZ2UnIH0pO1xuICAgICAgcGlwZWxpbmUuYWRkU3RhZ2Uoe1xuICAgICAgICBzdGFnZU5hbWU6ICdTZWNvbmRTdGFnZScsXG4gICAgICAgIHBsYWNlbWVudDoge1xuICAgICAgICAgIGp1c3RBZnRlcjogZmlyc3RTdGFnZSxcbiAgICAgICAgfSxcbiAgICAgIH0pO1xuXG4gICAgICBleHBlY3Qoc3RhY2ssIHRydWUpLnRvKGhhdmVSZXNvdXJjZUxpa2UoJ0FXUzo6Q29kZVBpcGVsaW5lOjpQaXBlbGluZScsIHtcbiAgICAgICAgJ1N0YWdlcyc6IFtcbiAgICAgICAgICB7ICdOYW1lJzogJ0ZpcnN0U3RhZ2UnIH0sXG4gICAgICAgICAgeyAnTmFtZSc6ICdTZWNvbmRTdGFnZScgfSxcbiAgICAgICAgICB7ICdOYW1lJzogJ1RoaXJkU3RhZ2UnIH0sXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICBcImF0dGVtcHRpbmcgdG8gaW5zZXJ0IGEgU3RhZ2UgYmVmb3JlIGEgU3RhZ2UgdGhhdCBkb2Vzbid0IGV4aXN0IHJlc3VsdHMgaW4gYW4gZXJyb3JcIih0ZXN0OiBUZXN0KSB7XG4gICAgICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgICAgIGNvbnN0IHBpcGVsaW5lID0gbmV3IGNvZGVwaXBlbGluZS5QaXBlbGluZShzdGFjaywgJ1BpcGVsaW5lJyk7XG4gICAgICBjb25zdCBzdGFnZSA9IHBpcGVsaW5lLmFkZFN0YWdlKHsgc3RhZ2VOYW1lOiAnU3RhZ2UnIH0pO1xuXG4gICAgICBjb25zdCBhbm90aGVyUGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHN0YWNrLCAnQW5vdGhlclBpcGVsaW5lJyk7XG4gICAgICB0ZXN0LnRocm93cygoKSA9PiB7XG4gICAgICAgIGFub3RoZXJQaXBlbGluZS5hZGRTdGFnZSh7XG4gICAgICAgICAgc3RhZ2VOYW1lOiAnQW5vdGhlclN0YWdlJyxcbiAgICAgICAgICBwbGFjZW1lbnQ6IHtcbiAgICAgICAgICAgIHJpZ2h0QmVmb3JlOiBzdGFnZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0sIC9iZWZvcmUvaSk7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG5cbiAgICBcImF0dGVtcHRpbmcgdG8gaW5zZXJ0IGEgU3RhZ2UgYWZ0ZXIgYSBTdGFnZSB0aGF0IGRvZXNuJ3QgZXhpc3QgcmVzdWx0cyBpbiBhbiBlcnJvclwiKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgcGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHN0YWNrLCAnUGlwZWxpbmUnKTtcbiAgICAgIGNvbnN0IHN0YWdlID0gcGlwZWxpbmUuYWRkU3RhZ2UoeyBzdGFnZU5hbWU6ICdTdGFnZScgfSk7XG5cbiAgICAgIGNvbnN0IGFub3RoZXJQaXBlbGluZSA9IG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUoc3RhY2ssICdBbm90aGVyUGlwZWxpbmUnKTtcbiAgICAgIHRlc3QudGhyb3dzKCgpID0+IHtcbiAgICAgICAgYW5vdGhlclBpcGVsaW5lLmFkZFN0YWdlKHtcbiAgICAgICAgICBzdGFnZU5hbWU6ICdBbm90aGVyU3RhZ2UnLFxuICAgICAgICAgIHBsYWNlbWVudDoge1xuICAgICAgICAgICAganVzdEFmdGVyOiBzdGFnZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIH0sIC9hZnRlci9pKTtcblxuICAgICAgdGVzdC5kb25lKCk7XG4gICAgfSxcblxuICAgICdwcm92aWRpbmcgbW9yZSB0aGFuIG9uZSBwbGFjZW1lbnQgdmFsdWUgcmVzdWx0cyBpbiBhbiBlcnJvcicodGVzdDogVGVzdCkge1xuICAgICAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gICAgICBjb25zdCBwaXBlbGluZSA9IG5ldyBjb2RlcGlwZWxpbmUuUGlwZWxpbmUoc3RhY2ssICdQaXBlbGluZScpO1xuICAgICAgY29uc3Qgc3RhZ2UgPSBwaXBlbGluZS5hZGRTdGFnZSh7IHN0YWdlTmFtZTogJ1N0YWdlJyB9KTtcblxuICAgICAgdGVzdC50aHJvd3MoKCkgPT4ge1xuICAgICAgICBwaXBlbGluZS5hZGRTdGFnZSh7XG4gICAgICAgICAgc3RhZ2VOYW1lOiAnU2Vjb25kU3RhZ2UnLFxuICAgICAgICAgIHBsYWNlbWVudDoge1xuICAgICAgICAgICAgcmlnaHRCZWZvcmU6IHN0YWdlLFxuICAgICAgICAgICAganVzdEFmdGVyOiBzdGFnZSxcbiAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgIC8vIGluY3JlZGlibHksIGFuIGFycm93IGZ1bmN0aW9uIGJlbG93IGNhdXNlcyBub2RldW5pdCB0byBjcmFwIG91dCB3aXRoOlxuICAgICAgLy8gXCJUeXBlRXJyb3I6IEZ1bmN0aW9uIGhhcyBub24tb2JqZWN0IHByb3RvdHlwZSAndW5kZWZpbmVkJyBpbiBpbnN0YW5jZW9mIGNoZWNrXCJcbiAgICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTpvbmx5LWFycm93LWZ1bmN0aW9uc1xuICAgICAgfSwgZnVuY3Rpb24oZTogYW55KSB7XG4gICAgICAgIHJldHVybiAvcmlnaHRCZWZvcmUvLnRlc3QoZSkgJiYgL2p1c3RBZnRlci8udGVzdChlKTtcbiAgICAgIH0pO1xuXG4gICAgICB0ZXN0LmRvbmUoKTtcbiAgICB9LFxuXG4gICAgJ2NhbiBiZSByZXRyaWV2ZWQgZnJvbSBhIHBpcGVsaW5lIGFmdGVyIGl0IGhhcyBiZWVuIGNyZWF0ZWQnKHRlc3Q6IFRlc3QpIHtcbiAgICAgIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICAgICAgY29uc3QgcGlwZWxpbmUgPSBuZXcgY29kZXBpcGVsaW5lLlBpcGVsaW5lKHN0YWNrLCAnUGlwZWxpbmUnLCB7XG4gICAgICAgIHN0YWdlczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIHN0YWdlTmFtZTogJ0ZpcnN0U3RhZ2UnLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9KTtcblxuICAgICAgcGlwZWxpbmUuYWRkU3RhZ2UoeyBzdGFnZU5hbWU6ICdTZWNvbmRTdGFnZScgfSk7XG5cbiAgICAgIHRlc3QuZXF1YWwocGlwZWxpbmUuc3RhZ2VzLmxlbmd0aCwgMik7XG4gICAgICB0ZXN0LmVxdWFsKHBpcGVsaW5lLnN0YWdlc1swXS5zdGFnZU5hbWUsICdGaXJzdFN0YWdlJyk7XG4gICAgICB0ZXN0LmVxdWFsKHBpcGVsaW5lLnN0YWdlc1sxXS5zdGFnZU5hbWUsICdTZWNvbmRTdGFnZScpO1xuXG4gICAgICAvLyBhZGRpbmcgc3RhZ2VzIHRvIHRoZSByZXR1cm5lZCBhcnJheSBzaG91bGQgaGF2ZSBubyBlZmZlY3RcbiAgICAgIHBpcGVsaW5lLnN0YWdlcy5wdXNoKG5ldyBTdGFnZSh7XG4gICAgICAgIHN0YWdlTmFtZTogJ1RoaXJkU3RhZ2UnLFxuICAgICAgfSwgcGlwZWxpbmUpKTtcbiAgICAgIHRlc3QuZXF1YWwocGlwZWxpbmUuc3RhZ2VDb3VudCwgMik7XG5cbiAgICAgIHRlc3QuZG9uZSgpO1xuICAgIH0sXG4gIH0sXG59O1xuIl19