# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/09_kernel.ipynb.

# %% auto 0
__all__ = ['PyStataKernel', 'Cell', 'print_stata_error']

# %% ../nbs/09_kernel.ipynb 4
from .config import get_config, launch_stata
from .helpers import get_inspect
from .utils import print_red, ending_delimiter, is_cr_delimiter
from .stata_session import StataSession
from .magics import StataMagics
from .completions import CompletionsManager
import nbstata # for __version__
from fastcore.basics import patch_to
from ipykernel.ipkernel import IPythonKernel

# %% ../nbs/09_kernel.ipynb 5
class PyStataKernel(IPythonKernel):
    """A jupyter kernel based on pystata"""
    implementation = 'nbstata'
    implementation_version = nbstata.__version__
    language_info = {
        'name': 'stata',
        'version': '17',
        'mimetype': 'text/x-stata',
        'file_extension': '.do',
    }
    banner = "nbstata: a Jupyter kernel for Stata based on pystata"
    help_links = [
        {
            "text": "Stata Documentation",
            "url": "https://www.stata.com/features/documentation/",
        },
        {
            "text": "nbstata Help",
            "url": "https://hugetim.github.io/nbstata/",
        },
    ]

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.stata_ready = False
        self.shell.execution_count = 0
        self.starting_delimiter = None
        self.perspective_enabled = None
        self.inspect_output = "Stata not yet initialized."
        try:
            self.init_stata()
        except ModuleNotFoundError as err:
            pass # wait for first do_execute so error message can be displayed under cell

# %% ../nbs/09_kernel.ipynb 6
def _set_graph_format(graph_format):
    if graph_format != 'pystata':
        from pystata.config import set_graph_format
        set_graph_format(graph_format)

# %% ../nbs/09_kernel.ipynb 7
def _config_stata(env):
    launch_stata(env['stata_dir'], env['edition'],
                 False if env['splash']=='False' else True)
    _set_graph_format(env['graph_format'])

# %% ../nbs/09_kernel.ipynb 8
@patch_to(PyStataKernel)
def init_stata(self):
    self.env = get_config()
    _config_stata(self.env)
    self.stata_session = StataSession()
    self.magic_handler = StataMagics()
    self.completions = CompletionsManager(self.stata_session, list(self.magic_handler.available_magics.keys()))
    self.inspect_output = ""
    self.stata_ready = True

# %% ../nbs/09_kernel.ipynb 9
class Cell:
    """A class for managing execution of a single code cell"""                
    def _set_echo(self, echo_config):
        if echo_config == 'None':
            self.noecho = True
            self.echo = False
        elif echo_config == 'True':
            self.noecho = False
            self.echo = True
        else:
            self.noecho = False
            self.echo = False
    
    def __init__(self, kernel, code_w_magics, silent=False):
        self._set_echo(kernel.env['echo'])
        self.quietly = silent
        self.delimiter = kernel.starting_delimiter
        self.stata_session = kernel.stata_session
        self.code = kernel.magic_handler.magic(code_w_magics, kernel, self)
 
    def _run_noecho(self):
        from nbstata.helpers import run_noecho
        run_noecho(self.code, self.delimiter,
                   run_as_prog=self.stata_session.run_as_prog_with_locals)

    def _run_simple(self):
        from pystata.stata import run
        if not is_cr_delimiter(self.delimiter):
            self.code = "#delimit;\n" + self.code
        run(self.code, quietly=self.quietly, inline=not self.quietly, echo=self.echo)
        
    def run(self):
        if not self.code:
            return
        if self.noecho and not self.quietly:
            self._run_noecho()
        else:
            self._run_simple()
        self.delimiter = self._check_ending_delimiter()

    def _check_ending_delimiter(self):
        _ending_delimiter = ending_delimiter(self.code, self.delimiter)
        _final_character = self.code.strip()[-1]
        _code_missing_final_delimiter = (_ending_delimiter == ';' 
                                         and _final_character != ';')
        if _code_missing_final_delimiter:
            print_red(
                self._final_delimiter_warning()
            )
        return _ending_delimiter
    
    def _final_delimiter_warning(self):
        return (
            "Warning: Code cell (with #delimit; in effect) does not end in ';'. "
            "Exported .do script may behave differently from notebook. "
            "In v1.0, nbstata may trigger an error instead of just a warning."
        )

# %% ../nbs/09_kernel.ipynb 21
_missing_stata_message = (
    "pystata path not found\n"
    "A Stata 17 installation is required to use the nbstata Stata kernel. "
    "If you already have Stata 17 installed, "
    "please specify its path in your configuration file."
)

# %% ../nbs/09_kernel.ipynb 23
def _handle_stata_import_error(err, silent, execution_count):
    if not silent:
        print_red(f"ModuleNotFoundError: {_missing_stata_message}")
    return {
        "traceback": [],
        "ename": "ModuleNotFoundError",
        "evalue": _missing_stata_message,
        'status': "error",
        'execution_count': execution_count,
    }

# %% ../nbs/09_kernel.ipynb 24
def print_stata_error(text):
    lines = text.splitlines()
    if len(lines) > 2:
        print("\n".join(lines[:-2]))
    print_red("\n".join(lines[-2:]))

# %% ../nbs/09_kernel.ipynb 26
def _handle_stata_error(err, silent, execution_count):
    reply_content = {
        "traceback": [],
        "ename": "Stata error",
        "evalue": str(err),
    }
    if not silent:
        print_stata_error(reply_content['evalue'])
#         self.send_response(
#             self.iopub_socket,
#             "error",
#             reply_content,
#         )
    reply_content.update({
        'status': "error",
        'execution_count': execution_count,
    })
    return reply_content

# %% ../nbs/09_kernel.ipynb 27
@patch_to(PyStataKernel)
def post_do_hook(self):
    self.stata_session.clear_suggestions()
    self.inspect_output = ""

# %% ../nbs/09_kernel.ipynb 28
@patch_to(PyStataKernel)
def do_execute(self, code, silent,
               store_history=True, user_expressions=None, allow_stdin=False):
    """Execute Stata code cell"""
    if not self.stata_ready:
        try:
            self.init_stata()
        except ModuleNotFoundError as err:
            return _handle_stata_import_error(err, silent, self.execution_count)
    self.shell.execution_count += 1
    code_cell = Cell(self, code, silent)
    try:
        code_cell.run()
    except SystemError as err:
        return _handle_stata_error(err, silent, self.execution_count)
    self.starting_delimiter = code_cell.delimiter
    self.post_do_hook()
    return {
        'status': "ok",
        'execution_count': self.execution_count,
        'payload': [],
        'user_expressions': {},
    }

# %% ../nbs/09_kernel.ipynb 29
@patch_to(PyStataKernel)
def do_complete(self, code, cursor_pos):
    """Provide context-aware suggestions"""
    if self.stata_ready:
        cursor_start, cursor_end, matches = self.completions.do(
            code,
            cursor_pos,
            self.starting_delimiter,
        )
    else:
        cursor_start = cursor_end = cursor_pos
        matches = []
    return {
        'status': "ok",
        'cursor_start': cursor_start,
        'cursor_end': cursor_end,
        'metadata': {},
        'matches': matches,
    }

# %% ../nbs/09_kernel.ipynb 30
@patch_to(PyStataKernel)
def do_is_complete(self, code):
    """Overrides IPythonKernel with kernelbase default"""
    return {"status": "unknown"}

# %% ../nbs/09_kernel.ipynb 31
@patch_to(PyStataKernel)
def do_inspect(self, code, cursor_pos, detail_level=0, omit_sections=()):
    """Display Stata 'describe' output regardless of cursor position"""
    if not self.inspect_output:
        self.inspect_output = get_inspect(code, cursor_pos, detail_level, omit_sections)
    data = {'text/plain': self.inspect_output}
    return {"status": "ok", "data": data, "metadata": {}, "found": True}

# %% ../nbs/09_kernel.ipynb 32
@patch_to(PyStataKernel)
def do_history(
    self,
    hist_access_type,
    output,
    raw,
    session=None,
    start=None,
    stop=None,
    n=None,
    pattern=None,
    unique=False,
):
    """Overrides IPythonKernel with kernelbase default"""
    return {"status": "ok", "history": []}
