"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const language_1 = require("./backend/shared/language");
const deep_link_1 = require("./deep-link");
const metric_utils_1 = require("./metric-utils");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new aws_cloudwatch_1.Dashboard(this, 'Resource', {
            dashboardName: props.dashboardName,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            start: '-P1W',
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Catalog Overview',
                            '',
                            `[button:primary:Package Data](${deep_link_1.s3ObjectUrl(props.packageData)})`,
                            `[button:Catalog Builder](${deep_link_1.lambdaFunctionUrl(props.orchestration.catalogBuilder)})`,
                            `[button:Inventory Canary](${deep_link_1.lambdaFunctionUrl(props.inventory.function)})`,
                            `[button:Search Canary Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.inventory.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props),
                ...renderPackageSourcesWidgets(props.packageSources),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Ingestion Function](${deep_link_1.lambdaFunctionUrl(props.ingestion.function)})`,
                            `[button:primary:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            metric_utils_1.fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            metric_utils_1.fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            metric_utils_1.fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:primary:State Machine](${deep_link_1.stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${deep_link_1.lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Reprocess](${deep_link_1.lambdaFunctionUrl(props.orchestration.reprocessAllFunction)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                // deny list
                // ----------------------------------------------
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Deny List',
                            '',
                            `[button:primary:Deny List Object](${deep_link_1.s3ObjectUrl(props.denyList.bucket, props.denyList.objectKey)})`,
                            `[button:Prune Function](${deep_link_1.lambdaFunctionUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Prune Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Delete Queue](${deep_link_1.sqsQueueUrl(props.denyList.prune.queue)})`,
                            `[button:Delete Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.deleteHandler)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Deny List',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.metricDenyListRules({ label: 'Rules' }), 'REPEAT'),
                            props.denyList.prune.queue.metricNumberOfMessagesDeleted({ label: 'Deleted Files' }),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Prune Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                ...(props.packageStats ? renderPackageStatsWidgets(props.packageStats) : []),
            ],
        });
    }
    *catalogOverviewLanguageSections({ inventory, orchestration }) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Documentation Generation',
                    '',
                    `[button:primary:Transliterator Logs](${deep_link_1.logGroupUrl(orchestration.transliterator.logGroup)})`,
                    `[button:Transliterator ECS Cluster](${deep_link_1.ecsClusterUrl(orchestration.ecsCluster)})`,
                ].join('\n'),
            }),
        ];
        const mFargateUsage = new aws_cloudwatch_1.Metric({
            dimensionsMap: {
                Class: 'None',
                Resource: 'OnDemand',
                Service: 'Fargate',
                Type: 'Resource',
            },
            metricName: 'ResourceCount',
            namespace: 'AWS/Usage',
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
        });
        yield [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Fargate Resources',
                left: [
                    mFargateUsage.with({ label: 'Fargate Usage (On-Demand)' }),
                    new aws_cloudwatch_1.MathExpression({
                        expression: 'SERVICE_QUOTA(mFargateUsage)',
                        label: 'Fargate Quota (On-Demand)',
                        usingMetrics: { mFargateUsage },
                    }),
                ],
                leftYAxis: { min: 0 },
                right: [
                    orchestration.metricEcsCpuUtilization({ label: 'CPU Utilization' }),
                    orchestration.metricEcsMemoryUtilization({ label: 'Memory Utilization' }),
                ],
                rightYAxis: { label: 'Percent', min: 0, max: 100, showUnits: false },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'ECS Resources',
                left: [
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkRxBytes({ label: 'Received Bytes' })),
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkTxBytes({ label: 'Transmitted Bytes' })),
                ],
                leftYAxis: { min: 0 },
                right: [
                    metric_utils_1.fillMetric(orchestration.metricEcsTaskCount({ label: 'Task Count' })),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function* renderPackageSourcesWidgets(packageSources) {
    var _a;
    for (const packageSource of packageSources) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    `# ${packageSource.name}`,
                    '',
                    ...((_a = packageSource.links) !== null && _a !== void 0 ? _a : [])
                        .map(({ name, primary, url }) => `[${primary ? 'button:primary' : 'button'}:${name}](${url})`),
                ].join('\n'),
            }),
        ];
        yield* packageSource.dashboardWidgets;
    }
    return;
}
function renderPackageStatsWidgets(packageStats) {
    return [
        [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Package Stats',
                    '',
                    `[button:primary:Package Stats Object](${deep_link_1.s3ObjectUrl(packageStats.bucket, packageStats.statsKey)})`,
                    `[button:Package Stats Function](${deep_link_1.lambdaFunctionUrl(packageStats.handler)})`,
                    `[button:Package Stats Logs](${deep_link_1.lambdaSearchLogGroupUrl(packageStats.handler)})`,
                ].join('\n'),
            }),
        ],
        [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Number of Package Stats Recorded',
                left: [
                    metric_utils_1.fillMetric(packageStats.metricPackagesCount({ label: 'Packages with stats' }), 'REPEAT'),
                ],
                leftYAxis: { min: 0 },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Invocation Duration',
                left: [
                    packageStats.handler.metricDuration({ label: 'Duration' }),
                ],
                leftYAxis: { min: 0 },
                rightAnnotations: [{
                        color: '#ffa500',
                        label: '15 minutes (Lambda timeout)',
                        value: core_1.Duration.minutes(15).toSeconds(),
                    }],
            }),
        ],
    ];
}
//# sourceMappingURL=data:application/json;base64,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