"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const https = require("https");
const url_1 = require("url");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const aws_lambda_shared_1 = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
/**
 * This function is invoked by the `npm-js-follower.lambda`  with a `PackageVersion` object, or by
 * an SQS trigger feeding from this function's Dead-Letter Queue (for re-trying purposes).
 *
 * The payload contains information about a discovered new package version. This handler will
 * check the package is not deny-listed, download the tarball from the payload's `tarballUrl`, store
 * it in the staging bucket, then send a message to the ConstructHub ingestion SQS queue in order to
 * trigger ingestion of this package.
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    if ('Records' in event) {
        // ATTENTION: Assumes there is only exactly 1 event in there...
        event = JSON.parse(event.Records[0].body);
        console.log(`Event (extracted): ${JSON.stringify(event, null, 2)}`);
    }
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const denyRule = denyList.lookup(event.name, event.version);
    if (denyRule != null) {
        console.log(`Package was deny-listed: ${denyRule.reason}`);
        return;
    }
    // Download the tarball
    console.log(`Downloading tarball from URL: ${event.tarballUrl}`);
    const tarball = await httpGet(event.tarballUrl);
    // Store the tarball into the staging bucket
    // - infos.dist.tarball => https://registry.npmjs.org/<@scope>/<name>/-/<name>-<version>.tgz
    // - stagingKey         =>                     staged/<@scope>/<name>/-/<name>-<version>.tgz
    const stagingKey = `${"staged/" /* STAGED_KEY_PREFIX */}${new url_1.URL(event.tarballUrl).pathname}`.replace(/\/{2,}/g, '/');
    console.log(`Storing tarball in staging bucket with key ${stagingKey}`);
    await aws_lambda_shared_1.s3().putObject({
        Bucket: stagingBucket,
        Key: stagingKey,
        Body: tarball,
        ContentType: 'application/octet-stream',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Modified-At': event.modified,
            'Origin-Integrity': event.integrity,
            'Origin-URI': event.tarballUrl,
            'Sequence': event.seq,
        },
    }).promise();
    // Prepare ingestion request
    const message = integrity_lambda_shared_1.integrity({
        tarballUri: `s3://${stagingBucket}/${stagingKey}`,
        metadata: {
            dist: event.tarballUrl,
            integrity: event.integrity,
            modified: event.modified,
            seq: event.seq,
        },
        time: event.modified,
    }, tarball);
    // Send message to SQS
    console.log(`Sending message to ConstructHub ingestion queue: ${JSON.stringify(message, null, 2)}`);
    await aws_lambda_shared_1.sqs().sendMessage({
        MessageBody: JSON.stringify(message, null, 2),
        MessageAttributes: {
            'Lambda-Log-Group': { DataType: 'String', StringValue: context.logGroupName },
            'Lambda-Log-Stream': { DataType: 'String', StringValue: context.logStreamName },
            'Lambda-Run-Id': { DataType: 'String', StringValue: context.awsRequestId },
        },
        QueueUrl: queueUrl,
    }).promise();
}
exports.handler = handler;
/**
 * Makes an HTTP GET request, and returns the resulting payload.
 *
 * @param url the URL to get.
 *
 * @returns a Buffer containing the received data.
 */
function httpGet(url) {
    return new Promise((ok, ko) => {
        https.get(url, (response) => {
            if (response.statusCode !== 200) {
                ko(new Error(`Unsuccessful GET: ${response.statusCode} - ${response.statusMessage}`));
            }
            let body = Buffer.alloc(0);
            response.on('data', (chunk) => body = Buffer.concat([body, Buffer.from(chunk)]));
            response.once('close', () => ok(body));
            response.once('error', ko);
        });
    });
}
//# sourceMappingURL=data:application/json;base64,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