"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const _lambda_architecture_1 = require("../_lambda-architecture");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const transliterator_1 = require("../transliterator");
const needs_catalog_update_1 = require("./needs-catalog-update");
const redrive_state_machine_1 = require("./redrive-state-machine");
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).next(new aws_stepfunctions_1.Succeed(this, 'Sent to DLQ'));
        const ignore = new aws_stepfunctions_1.Pass(this, 'Ignore');
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props);
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder.function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' });
        const needsCatalogUpdateFunction = new needs_catalog_update_1.NeedsCatalogUpdate(this, 'NeedsCatalogUpdate', {
            architecture: _lambda_architecture_1.gravitonLambdaIfAvailable(this),
            description: '[ConstructHub/Orchestration/NeedsCatalogUpdate] Determines whether a package version requires a catalog update',
            environment: { CATALOG_BUCKET_NAME: props.bucket.bucketName, CATALOG_OBJECT_KEY: constants_1.CATALOG_KEY },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(1),
        });
        props.bucket.grantRead(needsCatalogUpdateFunction);
        // Check whether the catalog needs updating. If so, trigger addToCatalog.
        const addToCatalogIfNeeded = new tasks.LambdaInvoke(this, 'Check whether catalog needs udpating', {
            lambdaFunction: needsCatalogUpdateFunction,
            payloadResponseOnly: true,
            resultPath: '$.catalogNeedsUpdating',
        })
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(new aws_stepfunctions_1.Choice(this, 'Is catalog update needed?')
            .when(aws_stepfunctions_1.Condition.booleanEquals('$.catalogNeedsUpdating', true), addToCatalog)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        })
            .next(new aws_stepfunctions_1.Pass(this, 'Prepare doc-gen ECS Command', {
            parameters: { 'command.$': 'States.Array(States.JsonToString($))' },
            resultPath: '$.docGen',
        }))
            .next(this.transliterator.createEcsRunTask(this, 'Generate docs', {
            cluster: this.ecsCluster,
            inputPath: '$.docGen.command',
            resultPath: '$.docGenOutput',
            // aws-cdk-lib succeeds in roughly 1 hour, so this should give us
            // enough of a buffer and prorably account for all other libraries out there.
            timeout: core_1.Duration.hours(2),
            vpcSubnets: props.vpcSubnets,
            securityGroups: props.vpcSecurityGroups,
        })
            // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
            .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
            .addRetry({
            errors: [
                'ECS.AmazonECSException',
                'ECS.InvalidParameterException',
                'jsii-docgen.NpmError.E429',
                'jsii-codgen.NpmError.EPROTO',
            ],
            ...THROTTLE_RETRY_POLICY,
        })
            .addRetry({
            errors: ['jsii-docgen.NpmError.ETARGET'],
            // We'll wait longer between retries. This is to account for CodeArtifact's lag behind npm
            backoffRate: 2,
            interval: core_1.Duration.minutes(5),
            maxAttempts: 3,
        })
            .addRetry({ maxAttempts: 3 })
            .addCatch(ignore, { errors: [constants_1.UNPROCESSABLE_PACKAGE_ERROR_NAME] })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.Timeout'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(addToCatalogIfNeeded));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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