"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const caching_1 = require("../../caching");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                     ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const created = new Array();
        const deleted = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'packages-'));
        const tarball = path.join(tmpdir, 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const uploads = new Map();
        const deletions = new Map();
        let unprocessable = false;
        function markPackage(e, marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const upload = uploadFile(event.bucket, key, event.assembly.versionId, Buffer.from(e.message));
            uploads.set(key, upload);
        }
        async function unmarkPackage(marker) {
            const key = event.assembly.key.replace(/\/[^/]+$/, marker);
            const marked = await aws.s3ObjectExists(event.bucket, key);
            if (!marked) {
                return;
            }
            const deletion = deleteFile(event.bucket, key);
            deletions.set(key, deletion);
        }
        console.log(`Generating documentation for ${packageFqn}...`);
        try {
            const docs = await docgen.Documentation.forPackage(tarball, { name: assembly.name });
            // if the package used to not be installabe, remove the marker for it.
            await unmarkPackage(constants.UNINSTALLABLE_PACKAGE_SUFFIX);
            for (const language of language_1.DocumentationLanguage.ALL) {
                if (event.languages && !event.languages[language.toString()]) {
                    console.log(`Skipping language ${language} as it was not requested!`);
                    continue;
                }
                const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
                    metrics.setDimensions();
                    metrics.setNamespace(constants_1.METRICS_NAMESPACE);
                    async function renderAndDispatch(submodule) {
                        try {
                            console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                            const markdown = await docs.render({
                                submodule,
                                linkFormatter: linkFormatter,
                                language: docgen.Language.fromString(lang.name),
                            });
                            // if the package used to have a corrupt assembly, remove the marker for it.
                            await unmarkPackage(constants.corruptAssemblyKeySuffix(language, submodule));
                            const page = Buffer.from(markdown.render());
                            metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                            const { buffer: body, contentEncoding } = compress_content_lambda_shared_1.compressContent(page);
                            metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                            const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule));
                            console.log(`Uploading ${key}`);
                            const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                            uploads.set(key, upload);
                        }
                        catch (e) {
                            if (e instanceof docgen.LanguageNotSupportedError) {
                                markPackage(e, constants.notSupportedKeySuffix(language, submodule));
                            }
                            else if (e instanceof docgen.CorruptedAssemblyError) {
                                markPackage(e, constants.corruptAssemblyKeySuffix(language, submodule));
                                unprocessable = true;
                            }
                            else {
                                throw e;
                            }
                        }
                    }
                    await renderAndDispatch();
                    for (const submodule of submodules) {
                        await renderAndDispatch(submodule);
                    }
                });
                await generateDocs(language);
            }
        }
        catch (error) {
            if (error instanceof docgen.UnInstallablePackageError) {
                markPackage(error, constants.UNINSTALLABLE_PACKAGE_SUFFIX);
                unprocessable = true;
            }
            else {
                throw error;
            }
        }
        for (const [key, upload] of uploads.entries()) {
            const response = await upload;
            created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
        }
        for (const [key, deletion] of deletions.entries()) {
            const response = await deletion;
            deleted.push({ bucket: event.bucket, key, versionId: response.VersionId });
            console.log(`Finished deleting ${key} (Version ID: ${response.VersionId})`);
        }
        if (unprocessable) {
            // the message here doesn't matter, we only use the error name
            // to divert this message away from the DLQ.
            const error = new Error();
            error.name = constants.UNPROCESSABLE_PACKAGE_ERROR_NAME;
        }
        return { created, deleted };
    });
}
exports.handler = handler;
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: caching_1.CacheStrategy.default().toString(),
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
function deleteFile(bucket, key) {
    return aws.s3().deleteObject({
        Bucket: bucket,
        Key: key,
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(type) {
    const packageName = type.source.assembly.name;
    const packageVersion = type.source.assembly.version;
    // the webapp sanitizes anchors - so we need to as well when
    // linking to them.
    const hash = sanitize(type.fqn);
    const query = `?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}`;
    return `/packages/${packageName}/v/${packageVersion}/api/${hash}${query}`;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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