"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get subProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.synthesizeNonNodePackageJson();
        super.synth();
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.subProjects.map((subProject) => path.relative(this.outdir, subProject.outdir)),
                nohoist: this.workspaceConfig?.noHoist,
            });
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.10.3" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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