"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
function deployNewFunc(stack) {
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain'
    };
    return new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
}
test('check domain names', () => {
    const stack = new cdk.Stack();
    deployNewFunc(stack);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain",
        UserPoolId: {
            Ref: "testlambdaelasticsearchstackCognitoUserPool05D1387E"
        }
    });
    expect(stack).toHaveResource('AWS::Elasticsearch::Domain', {
        DomainName: "test-domain",
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewFunc(stack);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.elasticsearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
    expect(construct.elasticsearchRole).toBeDefined();
});
test('check exception for Missing existingObj from props for deploy = false', () => {
    const stack = new cdk.Stack();
    const props = {
        domainName: 'test-domain'
    };
    try {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('check properties with no CW Alarms ', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        createCloudWatchAlarms: false
    };
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    expect(construct.lambdaFunction).toBeDefined();
    expect(construct.elasticsearchDomain).toBeDefined();
    expect(construct.identityPool).toBeDefined();
    expect(construct.userPool).toBeDefined();
    expect(construct.userPoolClient).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeUndefined();
    expect(construct.elasticsearchRole).toBeDefined();
});
test('check lambda function custom environment variable', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        domainEndpointEnvironmentVariableName: 'CUSTOM_DOMAIN_ENDPOINT'
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    expect(stack).toHaveResource('AWS::Lambda::Function', {
        Handler: 'index.handler',
        Runtime: 'nodejs14.x',
        Environment: {
            Variables: {
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                CUSTOM_DOMAIN_ENDPOINT: {
                    'Fn::GetAtt': [
                        'testlambdaelasticsearchstackElasticsearchDomain2DE7011B',
                        'DomainEndpoint'
                    ]
                }
            }
        }
    });
});
test('check override cognito domain name with provided cognito domain name', () => {
    const stack = new cdk.Stack();
    const props = {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        cognitoDomainName: 'test-cognito-domain'
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-stack', props);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: 'test-cognito-domain'
    });
});
test("Test minimal deployment that deploys a VPC in 2 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {});
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaelasticsearchkibanastackReplaceDefaultSecurityGroupsecuritygroup4C50002B",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                }
            ],
        },
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test minimal deployment that deploys a VPC in 3 AZ without vpcProps", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        deployVpc: true,
    });
    expect(stack).toHaveResource("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "lambdaelasticsearchkibanastackReplaceDefaultSecurityGroupsecuritygroup4C50002B",
                        "GroupId",
                    ],
                },
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B",
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055",
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D",
                },
            ],
        },
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toHaveResource("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test("Test ES cluster deploy to 1 AZ when user set zoneAwarenessEnabled to false", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const esDomainProps = {
        elasticsearchClusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            zoneAwarenessEnabled: false,
            instanceCount: 3
        }
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        esDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 1
        }
    });
    expect(stack).toHaveResource("AWS::Elasticsearch::Domain", {
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 3,
            ZoneAwarenessEnabled: false,
        }
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                }
            ]
        }
    });
});
test("Test ES cluster deploy to 2 AZ when user set availabilityZoneCount to 2", () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const esDomainProps = {
        elasticsearchClusterConfig: {
            dedicatedMasterCount: 3,
            dedicatedMasterEnabled: true,
            instanceCount: 2,
            zoneAwarenessEnabled: true,
            zoneAwarenessConfig: {
                availabilityZoneCount: 2
            }
        }
    };
    new lib_1.LambdaToElasticSearchAndKibana(stack, "lambda-elasticsearch-kibana-stack", {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: 'test-domain',
        esDomainProps,
        deployVpc: true,
        vpcProps: {
            maxAzs: 2
        }
    });
    expect(stack).toHaveResource("AWS::Elasticsearch::Domain", {
        ElasticsearchClusterConfig: {
            DedicatedMasterCount: 3,
            DedicatedMasterEnabled: true,
            InstanceCount: 2,
            ZoneAwarenessConfig: {
                AvailabilityZoneCount: 2,
            },
            ZoneAwarenessEnabled: true,
        }
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
});
test('Test minimal deployment with an existing isolated VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = defaults.getTestVpc(stack, false, {
        vpcName: "existing-isolated-vpc-test"
    });
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        existingVpc: vpc
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "existing-isolated-vpc-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with an existing private VPC', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const vpc = new ec2.Vpc(stack, 'existing-private-vpc-test', {
        natGateways: 1,
        subnetConfiguration: [
            {
                cidrMask: 24,
                name: 'application',
                subnetType: ec2.SubnetType.PRIVATE_WITH_NAT,
            },
            {
                cidrMask: 24,
                name: "public",
                subnetType: ec2.SubnetType.PUBLIC,
            }
        ]
    });
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        existingVpc: vpc
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "Default/existing-private-vpc-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "existingprivatevpctestapplicationSubnet1Subnet1F7744F0"
                },
                {
                    Ref: "existingprivatevpctestapplicationSubnet2SubnetF7B713AD"
                },
                {
                    Ref: "existingprivatevpctestapplicationSubnet3SubnetA519E038"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test minimal deployment with VPC construct props', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const construct = new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
        lambdaFunctionProps: getDefaultTestLambdaProps(),
        domainName: "test",
        deployVpc: true,
        vpcProps: {
            vpcName: "vpc-props-test"
        }
    });
    expect(stack).toHaveResourceLike("AWS::EC2::VPC", {
        Tags: [
            {
                Key: "Name",
                Value: "vpc-props-test"
            }
        ]
    });
    expect(stack).toHaveResourceLike("AWS::Elasticsearch::Domain", {
        VPCOptions: {
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                },
                {
                    Ref: "VpcisolatedSubnet3Subnet44F2537D"
                }
            ]
        }
    });
    expect(stack).toCountResources("AWS::EC2::VPC", 1);
    expect(construct.vpc).toBeDefined();
});
test('Test error for vpcProps and undefined deployVpc prop', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            domainName: "test",
            vpcProps: {
                vpcName: "existing-vpc-test"
            }
        });
    };
    expect(app).toThrowError("Error - deployVpc must be true when defining vpcProps");
});
test('Test error for Lambda function VPC props', () => {
    const stack = new cdk.Stack();
    const vpc = defaults.getTestVpc(stack);
    const app = () => {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
            lambdaFunctionProps: defaults.consolidateProps(getDefaultTestLambdaProps(), { vpc }),
            domainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not Lambda function props");
});
test('Test error for Elasticsearch domain VPC props', () => {
    const stack = new cdk.Stack();
    const app = () => {
        new lib_1.LambdaToElasticSearchAndKibana(stack, 'test-lambda-elasticsearch-kibana', {
            lambdaFunctionProps: getDefaultTestLambdaProps(),
            esDomainProps: {
                vpcOptions: {
                    subnetIds: ['fake-ids'],
                    securityGroupIds: ['fake-sgs']
                }
            },
            domainName: "test",
            deployVpc: true,
        });
    };
    expect(app).toThrowError("Error - Define VPC using construct parameters not Elasticsearch props");
});
function getDefaultTestLambdaProps() {
    return {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: 'index.handler',
    };
}
//# sourceMappingURL=data:application/json;base64,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