"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const semver_1 = require("semver");
const client_lambda_shared_1 = require("../catalog-builder/client.lambda-shared");
const constants_1 = require("../shared/constants");
/**
 * This function checks whether the catalog object needs updating following the
 * ingestion of the provided package object. This is the case if the package is
 * representative of a new major version line, or if the package is newer than,
 * or at the same version than the one already in catalog for the same major
 * version line.
 *
 * This is used to reduce how many calls are made into the catalog builder
 * function, as that function runs with singleton concurrency. This ultimately
 * allows workflows to complete faster when they are targetting older versions
 * of packages.
 */
async function handler(event) {
    var _a;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const [, packageName, version] = (_a = constants_1.STORAGE_KEY_FORMAT_REGEX.exec(event.package.key)) !== null && _a !== void 0 ? _a : die(`Unexpected/invalid package key: ${event.package.key}`);
    const packageMajor = semver_1.major(version);
    try {
        const catalogClient = await client_lambda_shared_1.CatalogClient.newClient();
        const existingEntry = catalogClient.packages.find((pkg) => pkg.name === packageName && pkg.major === packageMajor);
        if (existingEntry == null) {
            return true;
        }
        return semver_1.compare(version, existingEntry.version) >= 0;
    }
    catch (e) {
        if (e instanceof client_lambda_shared_1.CatalogNotFoundError) {
            // before the very first package was added, or after the catalog was deleted for some reason.
            return true;
        }
        throw e;
    }
}
exports.handler = handler;
function die(message) {
    const error = new Error(message);
    Error.captureStackTrace(error, die);
    throw error;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoibmVlZHMtY2F0YWxvZy11cGRhdGUubGFtYmRhLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2JhY2tlbmQvb3JjaGVzdHJhdGlvbi9uZWVkcy1jYXRhbG9nLXVwZGF0ZS5sYW1iZGEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsbUNBQXdDO0FBQ3hDLGtGQUE4RjtBQUU5RixtREFBK0Q7QUFJL0Q7Ozs7Ozs7Ozs7O0dBV0c7QUFDSSxLQUFLLFVBQVUsT0FBTyxDQUFDLEtBQVk7O0lBQ3hDLE9BQU8sQ0FBQyxHQUFHLENBQUMsVUFBVSxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBRXhELE1BQU0sQ0FBQyxFQUFFLFdBQVcsRUFBRSxPQUFPLENBQUMsU0FBRyxvQ0FBd0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsbUNBQUksR0FBRyxDQUFDLG1DQUFtQyxLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsRUFBRSxDQUFDLENBQUM7SUFDakosTUFBTSxZQUFZLEdBQUcsY0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBRXBDLElBQUk7UUFDRixNQUFNLGFBQWEsR0FBRyxNQUFNLG9DQUFhLENBQUMsU0FBUyxFQUFFLENBQUM7UUFDdEQsTUFBTSxhQUFhLEdBQUcsYUFBYSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxHQUFHLEVBQUUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxJQUFJLEtBQUssV0FBVyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEtBQUssWUFBWSxDQUFDLENBQUM7UUFDbkgsSUFBSSxhQUFhLElBQUksSUFBSSxFQUFFO1lBQ3pCLE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFFRCxPQUFPLGdCQUFPLENBQUMsT0FBTyxFQUFFLGFBQWEsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7S0FDckQ7SUFBQyxPQUFPLENBQUMsRUFBRTtRQUNWLElBQUksQ0FBQyxZQUFZLDJDQUFvQixFQUFFO1lBQ3JDLDZGQUE2RjtZQUM3RixPQUFPLElBQUksQ0FBQztTQUNiO1FBQ0QsTUFBTSxDQUFDLENBQUM7S0FDVDtBQUVILENBQUM7QUF0QkQsMEJBc0JDO0FBRUQsU0FBUyxHQUFHLENBQUMsT0FBZTtJQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUNqQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLEdBQUcsQ0FBQyxDQUFDO0lBQ3BDLE1BQU0sS0FBSyxDQUFDO0FBQ2QsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGNvbXBhcmUsIG1ham9yIH0gZnJvbSAnc2VtdmVyJztcbmltcG9ydCB7IENhdGFsb2dDbGllbnQsIENhdGFsb2dOb3RGb3VuZEVycm9yIH0gZnJvbSAnLi4vY2F0YWxvZy1idWlsZGVyL2NsaWVudC5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB0eXBlIHsgU3RhdGVNYWNoaW5lSW5wdXQgfSBmcm9tICcuLi9wYXlsb2FkLXNjaGVtYSc7XG5pbXBvcnQgeyBTVE9SQUdFX0tFWV9GT1JNQVRfUkVHRVggfSBmcm9tICcuLi9zaGFyZWQvY29uc3RhbnRzJztcblxuZXhwb3J0IHR5cGUgSW5wdXQgPSBQaWNrPFN0YXRlTWFjaGluZUlucHV0LCAncGFja2FnZSc+O1xuXG4vKipcbiAqIFRoaXMgZnVuY3Rpb24gY2hlY2tzIHdoZXRoZXIgdGhlIGNhdGFsb2cgb2JqZWN0IG5lZWRzIHVwZGF0aW5nIGZvbGxvd2luZyB0aGVcbiAqIGluZ2VzdGlvbiBvZiB0aGUgcHJvdmlkZWQgcGFja2FnZSBvYmplY3QuIFRoaXMgaXMgdGhlIGNhc2UgaWYgdGhlIHBhY2thZ2UgaXNcbiAqIHJlcHJlc2VudGF0aXZlIG9mIGEgbmV3IG1ham9yIHZlcnNpb24gbGluZSwgb3IgaWYgdGhlIHBhY2thZ2UgaXMgbmV3ZXIgdGhhbixcbiAqIG9yIGF0IHRoZSBzYW1lIHZlcnNpb24gdGhhbiB0aGUgb25lIGFscmVhZHkgaW4gY2F0YWxvZyBmb3IgdGhlIHNhbWUgbWFqb3JcbiAqIHZlcnNpb24gbGluZS5cbiAqXG4gKiBUaGlzIGlzIHVzZWQgdG8gcmVkdWNlIGhvdyBtYW55IGNhbGxzIGFyZSBtYWRlIGludG8gdGhlIGNhdGFsb2cgYnVpbGRlclxuICogZnVuY3Rpb24sIGFzIHRoYXQgZnVuY3Rpb24gcnVucyB3aXRoIHNpbmdsZXRvbiBjb25jdXJyZW5jeS4gVGhpcyB1bHRpbWF0ZWx5XG4gKiBhbGxvd3Mgd29ya2Zsb3dzIHRvIGNvbXBsZXRlIGZhc3RlciB3aGVuIHRoZXkgYXJlIHRhcmdldHRpbmcgb2xkZXIgdmVyc2lvbnNcbiAqIG9mIHBhY2thZ2VzLlxuICovXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gaGFuZGxlcihldmVudDogSW5wdXQpOiBQcm9taXNlPGJvb2xlYW4+IHtcbiAgY29uc29sZS5sb2coYEV2ZW50OiAke0pTT04uc3RyaW5naWZ5KGV2ZW50LCBudWxsLCAyKX1gKTtcblxuICBjb25zdCBbLCBwYWNrYWdlTmFtZSwgdmVyc2lvbl0gPSBTVE9SQUdFX0tFWV9GT1JNQVRfUkVHRVguZXhlYyhldmVudC5wYWNrYWdlLmtleSkgPz8gZGllKGBVbmV4cGVjdGVkL2ludmFsaWQgcGFja2FnZSBrZXk6ICR7ZXZlbnQucGFja2FnZS5rZXl9YCk7XG4gIGNvbnN0IHBhY2thZ2VNYWpvciA9IG1ham9yKHZlcnNpb24pO1xuXG4gIHRyeSB7XG4gICAgY29uc3QgY2F0YWxvZ0NsaWVudCA9IGF3YWl0IENhdGFsb2dDbGllbnQubmV3Q2xpZW50KCk7XG4gICAgY29uc3QgZXhpc3RpbmdFbnRyeSA9IGNhdGFsb2dDbGllbnQucGFja2FnZXMuZmluZCgocGtnKSA9PiBwa2cubmFtZSA9PT0gcGFja2FnZU5hbWUgJiYgcGtnLm1ham9yID09PSBwYWNrYWdlTWFqb3IpO1xuICAgIGlmIChleGlzdGluZ0VudHJ5ID09IG51bGwpIHtcbiAgICAgIHJldHVybiB0cnVlO1xuICAgIH1cblxuICAgIHJldHVybiBjb21wYXJlKHZlcnNpb24sIGV4aXN0aW5nRW50cnkudmVyc2lvbikgPj0gMDtcbiAgfSBjYXRjaCAoZSkge1xuICAgIGlmIChlIGluc3RhbmNlb2YgQ2F0YWxvZ05vdEZvdW5kRXJyb3IpIHtcbiAgICAgIC8vIGJlZm9yZSB0aGUgdmVyeSBmaXJzdCBwYWNrYWdlIHdhcyBhZGRlZCwgb3IgYWZ0ZXIgdGhlIGNhdGFsb2cgd2FzIGRlbGV0ZWQgZm9yIHNvbWUgcmVhc29uLlxuICAgICAgcmV0dXJuIHRydWU7XG4gICAgfVxuICAgIHRocm93IGU7XG4gIH1cblxufVxuXG5mdW5jdGlvbiBkaWUobWVzc2FnZTogc3RyaW5nKTogbmV2ZXIge1xuICBjb25zdCBlcnJvciA9IG5ldyBFcnJvcihtZXNzYWdlKTtcbiAgRXJyb3IuY2FwdHVyZVN0YWNrVHJhY2UoZXJyb3IsIGRpZSk7XG4gIHRocm93IGVycm9yO1xufVxuIl19