import * as iam from '@aws-cdk/aws-iam';
import { RetentionDays } from '@aws-cdk/aws-logs';
import * as sqs from '@aws-cdk/aws-sqs';
import { Construct as CoreConstruct, Duration } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { AlarmActions, Domain } from './api';
import { DenyListRule } from './backend/deny-list/api';
import { DomainRedirectSource } from './domain-redirect';
import { IPackageSource } from './package-source';
import { PackageTag } from './package-tag';
import { PackageTagGroup } from './package-tag-group';
import { PreloadFile } from './preload-file';
import { SpdxLicense } from './spdx-license';
import { PackageLinkConfig, FeaturedPackages, FeatureFlags, Category } from './webapp';
/**
 * Props for `ConstructHub`.
 */
export interface ConstructHubProps {
    /**
     * Connect the hub to a domain (requires a hosted zone and a certificate).
     */
    readonly domain?: Domain;
    /**
     * Actions to perform when alarms are set.
     */
    readonly alarmActions?: AlarmActions;
    /**
     * Whether compute environments for sensitive tasks (which operate on
     * un-trusted complex data, such as the transliterator, which operates with
     * externally-sourced npm package tarballs) should run in network-isolated
     * environments. This implies the creation of additonal resources, including:
     *
     * - A VPC with only isolated subnets.
     * - VPC Endpoints (CloudWatch Logs, CodeArtifact, CodeArtifact API, S3, ...)
     * - A CodeArtifact Repository with an external connection to npmjs.com
     *
     * @deprecated use sensitiveTaskIsolation instead.
     */
    readonly isolateSensitiveTasks?: boolean;
    /**
     * Whether compute environments for sensitive tasks (which operate on
     * un-trusted complex data, such as the transliterator, which operates with
     * externally-sourced npm package tarballs) should run in network-isolated
     * environments. This implies the creation of additonal resources, including:
     *
     * - A VPC with only isolated subnets.
     * - VPC Endpoints (CloudWatch Logs, CodeArtifact, CodeArtifact API, S3, ...)
     * - A CodeArtifact Repository with an external connection to npmjs.com
     *
     * @default Isolation.NO_INTERNET_ACCESS
     */
    readonly sensitiveTaskIsolation?: Isolation;
    /**
     * How long to retain CloudWatch logs for.
     *
     * @defaults RetentionDays.TEN_YEARS
     */
    readonly logRetention?: RetentionDays;
    /**
     * The name of the CloudWatch dashboard that represents the health of backend
     * systems.
     */
    readonly backendDashboardName?: string;
    /**
     * A list of packages to block from the construct hub.
     *
     * @default []
     */
    readonly denyList?: DenyListRule[];
    /**
     * The package sources to register with this ConstructHub instance.
     *
     * @default - a standard npmjs.com package source will be configured.
     */
    readonly packageSources?: IPackageSource[];
    /**
     * The allowed licenses for packages indexed by this instance of ConstructHub.
     *
     * @default [...SpdxLicense.apache(),...SpdxLicense.bsd(),...SpdxLicense.cddl(),...SpdxLicense.epl(),SpdxLicense.ISC,...SpdxLicense.mit(),SpdxLicense.MPL_2_0]
     */
    readonly allowedLicenses?: SpdxLicense[];
    /**
     * When using a CodeArtifact package source, it is often desirable to have
     * ConstructHub provision it's internal CodeArtifact repository in the same
     * CodeArtifact domain, and to configure the package source repository as an
     * upstream of the internal repository. This way, all packages in the source
     * are available to ConstructHub's backend processing.
     *
     * @default - none.
     */
    readonly codeArtifactDomain?: CodeArtifactDomainProps;
    /**
     * Configuration for custom package page links.
     */
    readonly packageLinks?: PackageLinkConfig[];
    /**
     * Configuration for custom package tags
     */
    readonly packageTags?: PackageTag[];
    /**
     * Optional configuration for grouping custom package tags
     */
    readonly packageTagGroups?: PackageTagGroup[];
    /**
     * Configuration for packages to feature on the home page.
     * @default - Display the 10 most recently updated packages
     */
    readonly featuredPackages?: FeaturedPackages;
    /**
     * Configure feature flags for the web app.
     */
    readonly featureFlags?: FeatureFlags;
    /**
     * Configure whether or not the backend should periodically query NPM
     * for the number of downloads a package has in the past week, and
     * display download counts on the web app.
     *
     * @default - true if packageSources is not specified (the defaults are
     * used), false otherwise
     */
    readonly fetchPackageStats?: boolean;
    /**
     * Browse categories. Each category will appear in the home page as a button
     * with a link to the relevant search query.
     */
    readonly categories?: Category[];
    /**
     * Wire construct hub to use the failover storage buckets.
     *
     * Do not activate this property until you've populated your failover buckets
     * with the necessary data.
     *
     * @see https://github.com/cdklabs/construct-hub/blob/dev/docs/operator-runbook.md#storage-disaster
     * @default false
     */
    readonly failoverStorage?: boolean;
    /**
     * How frequently all packages should get fully reprocessed.
     *
     * See the operator runbook for more information about reprocessing.
     * @see https://github.com/cdklabs/construct-hub/blob/main/docs/operator-runbook.md
     *
     * @default - never
     */
    readonly reprocessFrequency?: Duration;
    /**
     * Additional domains which will be set up to redirect to the primary
     * construct hub domain.
     *
     * @default []
     */
    readonly additionalDomains?: DomainRedirectSource[];
    /**
     * Javascript to run on webapp before app loads
     */
    readonly preloadScript?: PreloadFile;
}
/**
 * Information pertaining to an existing CodeArtifact Domain.
 */
export interface CodeArtifactDomainProps {
    /**
     * The name of the CodeArtifact domain.
     */
    readonly name: string;
    /**
     * Any upstream repositories in this CodeArtifact domain that should be
     * configured on the internal CodeArtifact repository.
     */
    readonly upstreams?: string[];
}
/**
 * Construct Hub.
 */
export declare class ConstructHub extends CoreConstruct implements iam.IGrantable {
    private readonly ingestion;
    constructor(scope: Construct, id: string, props?: ConstructHubProps);
    get grantPrincipal(): iam.IPrincipal;
    get ingestionQueue(): sqs.IQueue;
    private createVpc;
}
/**
 * How possibly risky operations (such as doc-generation, which requires
 * installing the indexed packages in order to trans-literate sample code) are
 * isolated to mitigate possible arbitrary code execution vulnerabilities in and
 * around `npm install` or the transliterator's use of the TypeScript compiler.
 */
export declare enum Isolation {
    /**
     * No isolation is done whatsoever. The doc-generation process still is
     * provisioned with least-privilege permissions, but retains complete access
     * to internet.
     *
     * While this maximizes the chances of successfully installing packages (and
     * hence successfully generating documentation for those), it is also the
     * least secure mode of operation.
     *
     * We advise you only consider using this isolation mode if you are hosting a
     * ConstructHub instance that only indexes trusted packages (including
     * transitive dependencies).
     */
    UNLIMITED_INTERNET_ACCESS = 0,
    /**
     * The same protections as `UNLIMITED_INTERNET_ACCESS`, except outbound
     * internet connections are limited to IP address ranges corresponding to
     * hosting endpoints for npmjs.com.
     */
    LIMITED_INTERNET_ACCESS = 1,
    /**
     * The same protections as `LIMITED_INTERNET_ACCESS`, except all remaining
     * internet access is removed. All traffic to AWS service endpoints is routed
     * through VPC Endpoints, as the compute nodes are jailed in a completely
     * isolated VPC.
     *
     * This is the most secure (and recommended) mode of operation for
     * ConstructHub instances.
     */
    NO_INTERNET_ACCESS = 2
}
