/// <reference types="node" />
import { EventEmitter } from 'events';
/**
 * A utility class that helps with traversing CouchDB database changes streams
 * in a promise-based, page-by-page manner.
 */
export declare class CouchChanges extends EventEmitter {
    private readonly agent;
    private readonly baseUrl;
    /**
     * @param baseUrl  the CouchDB endpoint URL.
     * @param database the name of the database for which changes are fetched.
     */
    constructor(baseUrl: string, database: string);
    /**
     * @returns summary informations about the database.
     */
    info(): Promise<DatabaseInfos>;
    /**
     * Obtains a batch of changes from the database.
     *
     * @param since     the sequence value since when history should be fetched.
     * @param batchSize the maximum amount of changes to return in a single page.
     *
     * @returns a page of changes.
     */
    changes(since: string | number, opts?: {
        readonly batchSize?: number;
    }): Promise<DatabaseChanges>;
    /**
     * Makes an HTTPs request using the provided method, url, and optionally payload. This function
     * properly handles input that is received with `Content-Type: gzip` and automatically retries
     * typical transient errors (HTTP 5XX, ECONNRESET, etc...) with linear back-off and no maximum
     * retry count (this is used in Lambda functions, which de-facto caps the amount of attempts
     * that will be made due to the function time out).
     *
     * @param method the HTTP method used for the request (e.g: 'get', 'post', ...).
     * @param url    the URL to request.
     * @param body   an optional HTTP request payload, which will be JSON-encoded.
     *
     * @param attempt the request attempt number (used to determine back-off / retry).
     *
     * @returns the JSON-decoded response body.
     */
    private https;
}
export interface DatabaseChanges {
    /**
     * The last sequence ID from this change set. This is the value that should be
     * passed to the subsequent `.changes` call to fetch the next page.
     */
    readonly last_seq: string | number;
    /**
     * The amount of pending changes from the server. This value is not always
     * returned by the servers.
     */
    readonly pending?: number;
    /**
     * The changes that are part of this batch.
     */
    readonly results: readonly DatabaseChange[];
}
export interface DatabaseChange {
    /**
     * The set of revisions to the object that were resolved as part of this
     * change.
     */
    readonly changes: ReadonlyArray<{
        readonly rev: string;
    }>;
    /**
     * The ID of the document that has changed.
     */
    readonly id: string;
    /**
     * The sequence ID for this change in the stream. It may not be present for
     * all (or any) entries in the result.
     */
    readonly seq?: string | number;
    /**
     * Whether this change corresponds to this document being deleted.
     */
    readonly deleted: boolean;
    /**
     * If present, the resolved document after the change has been applied.
     */
    readonly doc?: {
        readonly [key: string]: unknown;
    };
}
export interface DatabaseInfos {
    readonly db_name: string;
    readonly disk_format_version: number;
    readonly doc_count: number;
    readonly doc_del_count: number;
    readonly instance_start_time: string;
    readonly purge_seq: string | number;
    readonly sizes: {
        readonly active: number;
        readonly external: number;
        readonly file: number;
    };
    readonly update_seq: string | number;
}
