"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CouchChanges = void 0;
const events_1 = require("events");
const https_1 = require("https");
const url_1 = require("url");
const zlib_1 = require("zlib");
const JSONStream = require("JSONStream");
/**
 * A utility class that helps with traversing CouchDB database changes streams
 * in a promise-based, page-by-page manner.
 */
class CouchChanges extends events_1.EventEmitter {
    /**
     * @param baseUrl  the CouchDB endpoint URL.
     * @param database the name of the database for which changes are fetched.
     */
    constructor(baseUrl, database) {
        super();
        // Setting up for keep-alive connections.
        this.agent = new https_1.Agent({
            keepAlive: true,
            keepAliveMsecs: 5000,
            maxSockets: 4,
            timeout: 60000,
        });
        this.baseUrl = new url_1.URL(database, baseUrl);
    }
    /**
     * @returns summary informations about the database.
     */
    async info() {
        return await this.https('get', this.baseUrl);
    }
    /**
     * Obtains a batch of changes from the database.
     *
     * @param since     the sequence value since when history should be fetched.
     * @param batchSize the maximum amount of changes to return in a single page.
     *
     * @returns a page of changes.
     */
    async changes(since, opts) {
        var _a;
        const batchSize = (_a = opts === null || opts === void 0 ? void 0 : opts.batchSize) !== null && _a !== void 0 ? _a : 100;
        const changesUrl = new url_1.URL('_changes', this.baseUrl);
        changesUrl.searchParams.set('include_docs', 'true');
        changesUrl.searchParams.set('limit', batchSize.toFixed());
        changesUrl.searchParams.set('selector', '_filter');
        changesUrl.searchParams.set('seq_interval', batchSize.toFixed());
        changesUrl.searchParams.set('since', since.toString());
        changesUrl.searchParams.set('timeout', '20000' /* ms */);
        const filter = { name: { $gt: null } };
        return this.https('post', changesUrl, filter);
    }
    /**
     * Makes an HTTPs request using the provided method, url, and optionally payload. This function
     * properly handles input that is received with `Content-Type: gzip` and automatically retries
     * typical transient errors (HTTP 5XX, ECONNRESET, etc...) with linear back-off and no maximum
     * retry count (this is used in Lambda functions, which de-facto caps the amount of attempts
     * that will be made due to the function time out).
     *
     * @param method the HTTP method used for the request (e.g: 'get', 'post', ...).
     * @param url    the URL to request.
     * @param body   an optional HTTP request payload, which will be JSON-encoded.
     *
     * @param attempt the request attempt number (used to determine back-off / retry).
     *
     * @returns the JSON-decoded response body.
     */
    https(method, url, body, attempt = 1) {
        return new Promise((ok, ko) => {
            const retry = () => setTimeout(() => {
                console.log(`Retrying ${method.toUpperCase()} ${url}`);
                this.https(method, url, body, attempt + 1).then(ok, ko);
            }, Math.min(500 * attempt, 5000));
            const headers = {
                'Accept': 'application/json',
                'Accept-Encoding': 'gzip',
            };
            if (body) {
                headers['Content-Type'] = 'application/json';
            }
            console.log(`Request: ${method.toUpperCase()} ${url}`);
            const req = https_1.request(url, {
                agent: this.agent,
                headers,
                method,
                port: 443,
                servername: url.hostname,
            }, (res) => {
                if (res.statusCode == null) {
                    const error = new Error(`[FATAL] Request failed: ${method.toUpperCase()} ${url}`);
                    Error.captureStackTrace(error);
                    return ko(error);
                }
                console.log(`Response: ${method.toUpperCase()} ${url} => HTTP ${res.statusCode} (${res.statusMessage})`);
                // Transient (server) errors:
                if (res.statusCode >= 500 && res.statusCode < 600) {
                    console.error(`[RETRYABLE] HTTP ${res.statusCode} (${res.statusMessage}) - ${method.toUpperCase()} ${url}`);
                    // Call again after a short back-off
                    return retry();
                }
                // Permanent (client) errors:
                if (res.statusCode >= 400 && res.statusCode < 500) {
                    const error = new Error(`[FATAL] HTTP ${res.statusCode} (${res.statusMessage}) - ${method.toUpperCase()} ${url}`);
                    Error.captureStackTrace(error);
                    return ko(error);
                }
                const onError = (err) => {
                    if (err.code === 'ECONNRESET') {
                        // Transient networking problem?
                        console.error(`[RETRYABLE] ${err.code} - ${method.toUpperCase()} ${url}`);
                        retry();
                    }
                    else {
                        ko(err);
                    }
                };
                res.once('error', onError);
                const json = JSONStream.parse(true);
                json.once('data', ok);
                json.once('error', onError);
                const plainPayload = res.headers['content-encoding'] === 'gzip' ? gunzip(res) : res;
                plainPayload.pipe(json, { end: true });
            });
            req.end(body && JSON.stringify(body, null, 2));
        });
    }
}
exports.CouchChanges = CouchChanges;
function gunzip(readable) {
    const gz = zlib_1.createGunzip();
    readable.pipe(gz, { end: true });
    return gz;
}
//# sourceMappingURL=data:application/json;base64,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