"""Ramp utility module."""

# nominal strength values
BO_INJECTION_ENERGY = 0.150  # [GeV]
BO_EJECTION_ENERGY = 3.0  # [GeV]

BO_INJECTION_RF_GAPVOLT = 150.0074530757551  # [kV]
BO_EJECTION_RF_GAPVOLT = 950.0  # [kV]

NOMINAL_STRENGTHS = {
    'SI-Fam:PS-B1B2-1': BO_EJECTION_ENERGY,  # [Energy: GeV]
    'SI-Fam:PS-B1B2-2': BO_EJECTION_ENERGY,  # [Energy: GeV]
    'SI-Fam:PS-QFA': +0.7146305692912001,  # [KL: 1/m]
    'SI-Fam:PS-QDA': -0.2270152048045000,  # [KL: 1/m]
    'SI-Fam:PS-QFB': +1.2344424683922000,  # [KL: 1/m]
    'SI-Fam:PS-QDB2': -0.4782973132726601,  # [KL: 1/m]
    'SI-Fam:PS-QDB1': -0.2808906119138000,  # [KL: 1/m]
    'SI-Fam:PS-QFP': +1.2344424683922000,  # [KL: 1/m]
    'SI-Fam:PS-QDP2': -0.4782973132726601,  # [KL: 1/m]
    'SI-Fam:PS-QDP1': -0.2808906119138000,  # [KL: 1/m]
    'SI-Fam:PS-Q1': +0.5631612043340000,  # [KL: 1/m]
    'SI-Fam:PS-Q2': +0.8684629376249999,  # [KL: 1/m]
    'SI-Fam:PS-Q3': +0.6471254242426001,  # [KL: 1/m]
    'SI-Fam:PS-Q4': +0.7867827142062001,  # [KL: 1/m]
    'SI-Fam:PS-SDA0': -12.1250549999999979,  # [SL: 1/m^2]
    'SI-Fam:PS-SDB0': -09.7413299999999996,  # [SL: 1/m^2]
    'SI-Fam:PS-SDP0': -09.7413299999999996,  # [SL: 1/m^2]
    'SI-Fam:PS-SDA1': -24.4479749999999996,  # [SL: 1/m^2]
    'SI-Fam:PS-SDB1': -21.2453849999999989,  # [SL: 1/m^2]
    'SI-Fam:PS-SDP1': -21.3459000000000003,  # [SL: 1/m^2]
    'SI-Fam:PS-SDA2': -13.3280999999999992,  # [SL: 1/m^2]
    'SI-Fam:PS-SDB2': -18.3342150000000004,  # [SL: 1/m^2]
    'SI-Fam:PS-SDP2': -18.3421500000000002,  # [SL: 1/m^2]
    'SI-Fam:PS-SDA3': -20.9911199999999987,  # [SL: 1/m^2]
    'SI-Fam:PS-SDB3': -26.0718599999999974,  # [SL: 1/m^2]
    'SI-Fam:PS-SDP3': -26.1236099999999993,  # [SL: 1/m^2]
    'SI-Fam:PS-SFA0': +7.8854400000000000,  # [SL: 1/m^2]
    'SI-Fam:PS-SFB0': +11.0610149999999994,  # [SL: 1/m^2]
    'SI-Fam:PS-SFP0': +11.0610149999999994,  # [SL: 1/m^2]
    'SI-Fam:PS-SFA1': +28.7742599999999982,  # [SL: 1/m^2]
    'SI-Fam:PS-SFB1': +34.1821950000000001,  # [SL: 1/m^2]
    'SI-Fam:PS-SFP1': +34.3873949999999979,  # [SL: 1/m^2]
    'SI-Fam:PS-SFA2': +22.6153800000000018,  # [SL: 1/m^2]
    'SI-Fam:PS-SFB2': +29.6730900000000020,  # [SL: 1/m^2]
    'SI-Fam:PS-SFP2': +29.7755099999999970,  # [SL: 1/m^2]
    'BO-Fam:PS-B-1': BO_INJECTION_ENERGY,  # [Energy: GeV]
    'BO-Fam:PS-B-2': BO_INJECTION_ENERGY,  # [Energy: GeV]
    'BO-Fam:PS-QD': +0.0011197961538728,  # [KL: 1/m]
    'BO-Fam:PS-QF': +0.3770999232791374,  # [KL: 1/m]
    'BO-Fam:PS-SD': +0.5258382119529604,  # [SL: 1/m^2]
    'BO-Fam:PS-SF': +1.1898514030258744,  # [SL: 1/m^2]
}

# Power supply ramp parameters
DEFAULT_PS_RAMP_DURATION = 490.0  # [ms]

DEFAULT_PS_RAMP_RAMPUP1_START_TIME = 10.0  # [ms]
DEFAULT_PS_RAMP_RAMPUP2_START_TIME = 140.0  # [ms]
DEFAULT_PS_RAMP_RAMPDOWN_START_TIME = 378.0  # [ms]
DEFAULT_PS_RAMP_RAMPDOWN_STOP_TIME = 488.0  # [ms]
DEFAULT_PS_RAMP_RAMPUP_SMOOTH_INTVL = 60.0  # [ms]
DEFAULT_PS_RAMP_RAMPDOWN_SMOOTH_INTVL = 60.0  # [ms]

# --- Current values [A] ---
DEFAULT_PS_RAMP_START_CURRENT = 35.0
DEFAULT_PS_RAMP_RAMPUP1_START_CURRENT = 42.0
DEFAULT_PS_RAMP_RAMPUP2_START_CURRENT = 199.7
DEFAULT_PS_RAMP_RAMPDOWN_START_CURRENT = 1140.7
DEFAULT_PS_RAMP_RAMPDOWN_STOP_CURRENT = 42.0
DEFAULT_PS_RAMP_RAMPUP_SMOOTH_CURRENT = 5.7
DEFAULT_PS_RAMP_RAMPDOWN_SMOOTH_CURRENT = 91.3

# --- Energy values [GeV] ---
# NOTE: these constants are derived from excitation data
# current 150mev = 50.625127792833354
# current 3gev = 1013.2431132714938
DEFAULT_PS_RAMP_START_ENERGY = 0.10334265034313583
DEFAULT_PS_RAMP_RAMPUP1_START_ENERGY = 0.12409653823817533
DEFAULT_PS_RAMP_RAMPUP2_START_ENERGY = 0.5994074098589306
DEFAULT_PS_RAMP_RAMPDOWN_START_ENERGY = 3.328600085387852
DEFAULT_PS_RAMP_RAMPDOWN_STOP_ENERGY = 0.12409653823817533
DEFAULT_PS_RAMP_RAMPUP_SMOOTH_ENERGY = 0.01683008877016784
DEFAULT_PS_RAMP_RAMPDOWN_SMOOTH_ENERGY = 0.27256207120794684


# Timing parameters
DEFAULT_TI_PARAMS_INJECTION_TIME = 17.11  # [ms]
DEFAULT_TI_PARAMS_EJECTION_TIME = 345.76  # [ms]

DEFAULT_TI_PARAMS_PS_RAMP_DELAY = 0.0  # [us]
DEFAULT_TI_PARAMS_RF_RAMP_DELAY = DEFAULT_PS_RAMP_RAMPUP1_START_TIME  # [us]


# RF ramp parameters
MAX_RF_RAMP_DURATION = 500.0  # [ms]
DEFAULT_RF_RAMP_BOTTOM_DURATION = 0.0  # [ms]
DEFAULT_RF_RAMP_RAMPUP_DURATION = 340.0  # [ms]
DEFAULT_RF_RAMP_TOP_DURATION = 10.0  # [ms]
DEFAULT_RF_RAMP_RAMPDOWN_DURATION = 55.0  # [ms]
_DURATION = DEFAULT_RF_RAMP_BOTTOM_DURATION + \
            DEFAULT_RF_RAMP_RAMPUP_DURATION + \
            DEFAULT_RF_RAMP_TOP_DURATION + \
            DEFAULT_RF_RAMP_RAMPDOWN_DURATION
if _DURATION > MAX_RF_RAMP_DURATION:
    raise ValueError('Invalid RF ramp default durations.')
del _DURATION


def _calc_bot_top_voltage():
    # # Linear extrapolation to start and stop rf gap voltages
    _vt1 = DEFAULT_TI_PARAMS_INJECTION_TIME
    _vt2 = DEFAULT_TI_PARAMS_EJECTION_TIME
    _vv1 = BO_INJECTION_RF_GAPVOLT
    _vv2 = BO_EJECTION_RF_GAPVOLT
    _vmm = (_vv2 - _vv1)/(_vt2 - _vt1)
    _vtt = DEFAULT_TI_PARAMS_RF_RAMP_DELAY + DEFAULT_RF_RAMP_BOTTOM_DURATION
    def_bottom = _vv1 + _vmm*(_vtt - _vt1)  # [kV]
    _vtt = _vtt + DEFAULT_RF_RAMP_RAMPUP_DURATION
    def_top = _vv1 + _vmm*(_vtt - _vt1)  # [kV]
    return def_bottom, def_top


DEFAULT_RF_RAMP_BOTTOM_VOLTAGE, DEFAULT_RF_RAMP_TOP_VOLTAGE = \
    _calc_bot_top_voltage()

DEFAULT_RF_RAMP_BOTTOM_PHASE = 0.0  # [°]
DEFAULT_RF_RAMP_TOP_PHASE = 0.0  # [°]


def update_nominal_strengths(dic):
    """Update dictionary with nominal values."""
    for k, value in NOMINAL_STRENGTHS.items():
        if k in dic:
            dic[k] = value
