"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JsonFileSeeds = exports.InlineSeeds = exports.S3Seeds = exports.Seeds = void 0;
const fs = require("fs");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
class Seeds {
    /**
     * @returns `S3Seeds` associated with the specified S3 object.
     * @param bucket The S3 bucket
     * @param key The object key
     * @param objectVersion Optional S3 object version
     */
    static fromBucket(bucket, key, objectVersion) {
        return new S3Seeds(bucket, key, objectVersion);
    }
    /**
     * @returns `InlineSeeds` with inline seeds.
     * @param seeds The actual json code (limited to 4KiB)
     */
    static fromInline(seeds) {
        return new InlineSeeds(JSON.stringify(seeds));
    }
    /**
     * Loads the seeds from a local disk path and uploads it to s3.
     *
     * @returns `JsonFileSeeds` associated with the specified S3 object.
     * @param path Path to json seeds file.
     */
    static fromJsonFile(path, options) {
        return new JsonFileSeeds(path, options);
    }
}
exports.Seeds = Seeds;
/**
 * Seeds from an S3 archive.
 */
class S3Seeds extends Seeds {
    constructor(bucket, key, objectVersion) {
        super();
        this.key = key;
        this.objectVersion = objectVersion;
        if (!bucket.bucketName) {
            throw new Error('BucketName is undefined for the provided bucket');
        }
        this.bucketName = bucket.bucketName;
    }
    bind(_scope) {
        return {
            s3Location: {
                bucketName: this.bucketName,
                objectKey: this.key,
                objectVersion: this.objectVersion,
            },
        };
    }
}
exports.S3Seeds = S3Seeds;
/**
 * Seeds from an inline json object (limited to 4KiB).
 */
class InlineSeeds extends Seeds {
    constructor(seeds) {
        super();
        this.seeds = seeds;
        if (seeds.length === 0) {
            throw new Error('Inline seeds cannot be empty');
        }
        if (seeds.length > 4096) {
            throw new Error(`Inline seeds are too large, must be <= 4096 but is ${seeds.length}`);
        }
    }
    bind(_scope) {
        return {
            inlineSeeds: this.seeds,
        };
    }
}
exports.InlineSeeds = InlineSeeds;
/**
 * Seeds from a local json file.
 */
class JsonFileSeeds extends Seeds {
    constructor(path, options = {}) {
        super();
        this.path = path;
        this.options = options;
    }
    bind(scope) {
        // If the same JsonFileSeeds is used multiple times, retain only the first instantiation.
        if (!this.asset) {
            this.asset = new aws_s3_assets_1.Asset(scope, 'JsonFileSeeds', {
                path: this.path,
                ...this.options,
            });
        }
        try {
            JSON.parse(fs.readFileSync(this.path, 'utf-8'));
        }
        catch (e) {
            throw new Error('Could not convert file to JSON');
        }
        return {
            s3Location: {
                bucketName: this.asset.s3BucketName,
                objectKey: this.asset.s3ObjectKey,
            },
        };
    }
}
exports.JsonFileSeeds = JsonFileSeeds;
//# sourceMappingURL=data:application/json;base64,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