"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
            targetKey: encryptionKey,
            aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default');
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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