"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const action_1 = require("./action");
const codepipeline_generated_1 = require("./codepipeline.generated");
const cross_region_support_stack_1 = require("./cross-region-support-stack");
const full_action_descriptor_1 = require("./full-action-descriptor");
const stage_1 = require("./stage");
const validation_1 = require("./validation");
class PipelineBase extends core_1.Resource {
    /**
     * Defines an event rule triggered by this CodePipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.codepipeline'],
            resources: [this.pipelineArn],
        });
        return rule;
    }
    /**
     * Defines an event rule triggered by the "CodePipeline Pipeline Execution
     * State Change" event emitted from this pipeline.
     *
     * @param id Identifier for this event handler.
     * @param options Additional options to pass to the event rule.
     */
    onStateChange(id, options = {}) {
        const rule = this.onEvent(id, options);
        rule.addEventPattern({
            detailType: ['CodePipeline Pipeline Execution State Change'],
        });
        return rule;
    }
}
/**
 * An AWS CodePipeline pipeline with its associated IAM role and S3 bucket.
 *
 * @example
 * // create a pipeline
 * const pipeline = new Pipeline(this, 'Pipeline');
 *
 * // add a stage
 * const sourceStage = pipeline.addStage({ name: 'Source' });
 *
 * // add a source action to the stage
 * sourceStage.addAction(new codepipeline_actions.CodeCommitSourceAction({
 *   actionName: 'Source',
 *   outputArtifactName: 'SourceArtifact',
 *   repository: repo,
 * }));
 *
 * // ... add more stages
 */
class Pipeline extends PipelineBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.pipelineName,
        });
        this._stages = new Array();
        this._crossRegionSupport = {};
        this._crossAccountSupport = {};
        validation_1.validateName('Pipeline', this.physicalName);
        // only one of artifactBucket and crossRegionReplicationBuckets can be supplied
        if (props.artifactBucket && props.crossRegionReplicationBuckets) {
            throw new Error('Only one of artifactBucket and crossRegionReplicationBuckets can be specified!');
        }
        // If a bucket has been provided, use it - otherwise, create a bucket.
        let propsBucket = this.getArtifactBucketFromProps(props);
        if (!propsBucket) {
            const encryptionKey = new kms.Key(this, 'ArtifactsBucketEncryptionKey', {
                // remove the key - there is a grace period of a few days before it's gone for good,
                // that should be enough for any emergency access to the bucket artifacts
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
            propsBucket = new s3.Bucket(this, 'ArtifactsBucket', {
                bucketName: core_1.PhysicalName.GENERATE_IF_NEEDED,
                encryptionKey,
                encryption: s3.BucketEncryption.KMS,
                blockPublicAccess: new s3.BlockPublicAccess(s3.BlockPublicAccess.BLOCK_ALL),
                removalPolicy: core_1.RemovalPolicy.RETAIN,
            });
            // add an alias to make finding the key in the console easier
            new kms.Alias(this, 'ArtifactsBucketEncryptionKeyAlias', {
                aliasName: this.generateNameForDefaultBucketKeyAlias(),
                targetKey: encryptionKey,
                removalPolicy: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.artifactBucket = propsBucket;
        // If a role has been provided, use it - otherwise, create a role.
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codepipeline.amazonaws.com'),
        });
        const codePipeline = new codepipeline_generated_1.CfnPipeline(this, 'Resource', {
            artifactStore: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoreProperty() }),
            artifactStores: core_1.Lazy.anyValue({ produce: () => this.renderArtifactStoresProperty() }),
            stages: core_1.Lazy.anyValue({ produce: () => this.renderStages() }),
            roleArn: this.role.roleArn,
            restartExecutionOnUpdate: props && props.restartExecutionOnUpdate,
            name: this.physicalName,
        });
        // this will produce a DependsOn for both the role and the policy resources.
        codePipeline.node.addDependency(this.role);
        this.artifactBucket.grantReadWrite(this.role);
        this.pipelineName = this.getResourceNameAttribute(codePipeline.ref);
        this.pipelineVersion = codePipeline.attrVersion;
        this.crossRegionBucketsPassed = !!props.crossRegionReplicationBuckets;
        for (const [region, replicationBucket] of Object.entries(props.crossRegionReplicationBuckets || {})) {
            this._crossRegionSupport[region] = {
                replicationBucket,
                stack: core_1.Stack.of(replicationBucket),
            };
        }
        // Does not expose a Fn::GetAtt for the ARN so we'll have to make it ourselves
        this.pipelineArn = core_1.Stack.of(this).formatArn({
            service: 'codepipeline',
            resource: this.pipelineName,
        });
        for (const stage of props.stages || []) {
            this.addStage(stage);
        }
    }
    /**
     * Import a pipeline into this app.
     *
     * @param scope the scope into which to import this pipeline
     * @param id the logical ID of the returned pipeline construct
     * @param pipelineArn The ARN of the pipeline (e.g. `arn:aws:codepipeline:us-east-1:123456789012:MyDemoPipeline`)
     */
    static fromPipelineArn(scope, id, pipelineArn) {
        class Import extends PipelineBase {
            constructor() {
                super(...arguments);
                this.pipelineName = core_1.Stack.of(scope).parseArn(pipelineArn).resource;
                this.pipelineArn = pipelineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Creates a new Stage, and adds it to this Pipeline.
     *
     * @param props the creation properties of the new Stage
     * @returns the newly created Stage
     */
    addStage(props) {
        // check for duplicate Stages and names
        if (this._stages.find(s => s.stageName === props.stageName)) {
            throw new Error(`Stage with duplicate name '${props.stageName}' added to the Pipeline`);
        }
        const stage = new stage_1.Stage(props, this);
        const index = props.placement
            ? this.calculateInsertIndexFromPlacement(props.placement)
            : this.stageCount;
        this._stages.splice(index, 0, stage);
        return stage;
    }
    /**
     * Adds a statement to the pipeline role.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Get the number of Stages in this Pipeline.
     */
    get stageCount() {
        return this._stages.length;
    }
    /**
     * Returns the stages that comprise the pipeline.
     *
     * **Note**: the returned array is a defensive copy,
     * so adding elements to it has no effect.
     * Instead, use the {@link addStage} method if you want to add more stages
     * to the pipeline.
     */
    get stages() {
        return this._stages.slice();
    }
    /**
     * Returns all of the {@link CrossRegionSupportStack}s that were generated automatically
     * when dealing with Actions that reside in a different region than the Pipeline itself.
     *
     * @experimental
     */
    get crossRegionSupport() {
        const ret = {};
        Object.keys(this._crossRegionSupport).forEach((key) => {
            ret[key] = this._crossRegionSupport[key];
        });
        return ret;
    }
    /** @internal */
    _attachActionToPipeline(stage, action, actionScope) {
        // handle cross-region actions here
        const crossRegionInfo = this.ensureReplicationResourcesExistFor(action);
        // get the role for the given action
        const actionRole = this.getRoleForAction(stage, action, actionScope);
        // // CodePipeline Variables
        validation_1.validateNamespaceName(action.actionProperties.variablesNamespace);
        // bind the Action
        const actionConfig = action.bind(actionScope, stage, {
            role: actionRole ? actionRole : this.role,
            bucket: crossRegionInfo.artifactBucket,
        });
        return new full_action_descriptor_1.FullActionDescriptor({
            action,
            actionConfig,
            actionRole,
            actionRegion: crossRegionInfo.region,
        });
    }
    /**
     * Validate the pipeline structure
     *
     * Validation happens according to the rules documented at
     *
     * https://docs.aws.amazon.com/codepipeline/latest/userguide/reference-pipeline-structure.html#pipeline-requirements
     * @override
     */
    validate() {
        return [
            ...this.validateSourceActionLocations(),
            ...this.validateHasStages(),
            ...this.validateStages(),
            ...this.validateArtifacts(),
        ];
    }
    ensureReplicationResourcesExistFor(action) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRegion;
        let otherStack;
        const actionResource = action.actionProperties.resource;
        if (actionResource) {
            const actionResourceStack = core_1.Stack.of(actionResource);
            if (pipelineStack.region !== actionResourceStack.region) {
                actionRegion = actionResourceStack.region;
                // if the resource is from a different stack in another region but the same account,
                // use that stack as home for the cross-region support resources
                if (pipelineStack.account === actionResourceStack.account) {
                    otherStack = actionResourceStack;
                }
            }
        }
        else {
            actionRegion = action.actionProperties.region;
        }
        // if actionRegion is undefined,
        // it means the action is in the same region as the pipeline -
        // so, just return the artifactBucket
        if (!actionRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // get the region the Pipeline itself is in
        const pipelineRegion = this.requireRegion();
        // if the action is in the same region as the pipeline, nothing to do
        if (actionRegion === pipelineRegion) {
            return {
                artifactBucket: this.artifactBucket,
            };
        }
        // source actions have to be in the same region as the pipeline
        if (action.actionProperties.category === action_1.ActionCategory.SOURCE) {
            throw new Error(`Source action '${action.actionProperties.actionName}' must be in the same region as the pipeline`);
        }
        // check whether we already have a bucket in that region,
        // either passed from the outside or previously created
        let crossRegionSupport = this._crossRegionSupport[actionRegion];
        if (!crossRegionSupport) {
            // we need to create scaffolding resources for this region
            crossRegionSupport = this.createSupportResourcesForRegion(otherStack, actionRegion);
            this._crossRegionSupport[actionRegion] = crossRegionSupport;
        }
        // the stack containing the replication bucket must be deployed before the pipeline
        pipelineStack.addDependency(crossRegionSupport.stack);
        crossRegionSupport.replicationBucket.grantReadWrite(this.role);
        return {
            artifactBucket: crossRegionSupport.replicationBucket,
            region: actionRegion,
        };
    }
    createSupportResourcesForRegion(otherStack, actionRegion) {
        // if we have a stack from the resource passed - use that!
        if (otherStack) {
            // check if the stack doesn't have this magic construct already
            const id = `CrossRegionReplicationSupport-d823f1d8-a990-4e5c-be18-4ac698532e65-${actionRegion}`;
            let crossRegionSupportConstruct = otherStack.node.tryFindChild(id);
            if (!crossRegionSupportConstruct) {
                crossRegionSupportConstruct = new cross_region_support_stack_1.CrossRegionSupportConstruct(otherStack, id);
            }
            return {
                replicationBucket: crossRegionSupportConstruct.replicationBucket,
                stack: otherStack,
            };
        }
        // otherwise - create a stack with the resources needed for replication across regions
        const pipelineStack = core_1.Stack.of(this);
        const pipelineAccount = pipelineStack.account;
        if (core_1.Token.isUnresolved(pipelineAccount)) {
            throw new Error("You need to specify an explicit account when using CodePipeline's cross-region support");
        }
        const app = this.requireApp();
        const supportStackId = `cross-region-stack-${pipelineAccount}:${actionRegion}`;
        let supportStack = app.node.tryFindChild(supportStackId);
        if (!supportStack) {
            supportStack = new cross_region_support_stack_1.CrossRegionSupportStack(app, supportStackId, {
                pipelineStackName: pipelineStack.stackName,
                region: actionRegion,
                account: pipelineAccount,
                synthesizer: this.getCrossRegionSupportSynthesizer(),
            });
        }
        return {
            stack: supportStack,
            replicationBucket: supportStack.replicationBucket,
        };
    }
    getCrossRegionSupportSynthesizer() {
        if (this.stack.synthesizer instanceof core_1.DefaultStackSynthesizer) {
            // if we have the new synthesizer,
            // we need a bootstrapless copy of it,
            // because we don't want to require bootstrapping the environment
            // of the pipeline account in this replication region
            return new core_1.BootstraplessSynthesizer({
                deployRoleArn: this.stack.synthesizer.deployRoleArn,
                cloudFormationExecutionRoleArn: this.stack.synthesizer.cloudFormationExecutionRoleArn,
            });
        }
        else {
            // any other synthesizer: just return undefined
            // (ie., use the default based on the context settings)
            return undefined;
        }
    }
    generateNameForDefaultBucketKeyAlias() {
        const prefix = 'alias/codepipeline-';
        const maxAliasLength = 256;
        const uniqueId = this.node.uniqueId;
        // take the last 256 - (prefix length) characters of uniqueId
        const startIndex = Math.max(0, uniqueId.length - (maxAliasLength - prefix.length));
        return prefix + uniqueId.substring(startIndex).toLowerCase();
    }
    /**
     * Gets the role used for this action,
     * including handling the case when the action is supposed to be cross-account.
     *
     * @param stage the stage the action belongs to
     * @param action the action to return/create a role for
     * @param actionScope the scope, unique to the action, to create new resources in
     */
    getRoleForAction(stage, action, actionScope) {
        const pipelineStack = core_1.Stack.of(this);
        let actionRole = this.getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action);
        if (!actionRole && this.isAwsOwned(action)) {
            // generate a Role for this specific Action
            actionRole = new iam.Role(actionScope, 'CodePipelineActionRole', {
                assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            });
        }
        // the pipeline role needs assumeRole permissions to the action role
        if (actionRole) {
            this.role.addToPolicy(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                resources: [actionRole.roleArn],
            }));
        }
        return actionRole;
    }
    getRoleFromActionPropsOrGenerateIfCrossAccount(stage, action) {
        const pipelineStack = core_1.Stack.of(this);
        // if a Role has been passed explicitly, always use it
        // (even if the backing resource is from a different account -
        // this is how the user can override our default support logic)
        if (action.actionProperties.role) {
            if (this.isAwsOwned(action)) {
                // the role has to be deployed before the pipeline
                // (our magical cross-stack dependencies will not work,
                // because the role might be from a different environment),
                // but _only_ if it's a new Role -
                // an imported Role should not add the dependency
                if (action.actionProperties.role instanceof iam.Role) {
                    const roleStack = core_1.Stack.of(action.actionProperties.role);
                    pipelineStack.addDependency(roleStack);
                }
                return action.actionProperties.role;
            }
            else {
                // ...except if the Action is not owned by 'AWS',
                // as that would be rejected by CodePipeline at deploy time
                throw new Error("Specifying a Role is not supported for actions with an owner different than 'AWS' - " +
                    `got '${action.actionProperties.owner}' (Action: '${action.actionProperties.actionName}' in Stage: '${stage.stageName}')`);
            }
        }
        // if we don't have a Role passed,
        // and the action is cross-account,
        // generate a Role in that other account stack
        const otherAccountStack = this.getOtherStackIfActionIsCrossAccount(action);
        if (!otherAccountStack) {
            return undefined;
        }
        // if we have a cross-account action, the pipeline's bucket must have a KMS key
        if (!this.artifactBucket.encryptionKey) {
            throw new Error('The Pipeline is being used in a cross-account manner, ' +
                'but its artifact bucket does not have a KMS key defined. ' +
                'A KMS key is required for a cross-account Pipeline. ' +
                'Make sure to pass a Bucket with a Key when creating the Pipeline');
        }
        // generate a role in the other stack, that the Pipeline will assume for executing this action
        const ret = new iam.Role(otherAccountStack, `${this.node.uniqueId}-${stage.stageName}-${action.actionProperties.actionName}-ActionRole`, {
            assumedBy: new iam.AccountPrincipal(pipelineStack.account),
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
        });
        // the other stack with the role has to be deployed before the pipeline stack
        // (CodePipeline verifies you can assume the action Role on creation)
        pipelineStack.addDependency(otherAccountStack);
        return ret;
    }
    /**
     * Returns the Stack this Action belongs to if this is a cross-account Action.
     * If this Action is not cross-account (i.e., it lives in the same account as the Pipeline),
     * it returns undefined.
     *
     * @param action the Action to return the Stack for
     */
    getOtherStackIfActionIsCrossAccount(action) {
        const pipelineStack = core_1.Stack.of(this);
        if (action.actionProperties.resource) {
            const resourceStack = core_1.Stack.of(action.actionProperties.resource);
            // check if resource is from a different account
            if (pipelineStack.account === resourceStack.account) {
                return undefined;
            }
            else {
                this._crossAccountSupport[resourceStack.account] = resourceStack;
                return resourceStack;
            }
        }
        if (!action.actionProperties.account) {
            return undefined;
        }
        const targetAccount = action.actionProperties.account;
        // check whether the account is a static string
        if (core_1.Token.isUnresolved(targetAccount)) {
            throw new Error(`The 'account' property must be a concrete value (action: '${action.actionProperties.actionName}')`);
        }
        // check whether the pipeline account is a static string
        if (core_1.Token.isUnresolved(pipelineStack.account)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set account');
        }
        if (pipelineStack.account === targetAccount) {
            return undefined;
        }
        let targetAccountStack = this._crossAccountSupport[targetAccount];
        if (!targetAccountStack) {
            const stackId = `cross-account-support-stack-${targetAccount}`;
            const app = this.requireApp();
            targetAccountStack = app.node.tryFindChild(stackId);
            if (!targetAccountStack) {
                targetAccountStack = new core_1.Stack(app, stackId, {
                    stackName: `${pipelineStack.stackName}-support-${targetAccount}`,
                    env: {
                        account: targetAccount,
                        region: action.actionProperties.region ? action.actionProperties.region : pipelineStack.region,
                    },
                });
            }
            this._crossAccountSupport[targetAccount] = targetAccountStack;
        }
        return targetAccountStack;
    }
    isAwsOwned(action) {
        const owner = action.actionProperties.owner;
        return !owner || owner === 'AWS';
    }
    getArtifactBucketFromProps(props) {
        if (props.artifactBucket) {
            return props.artifactBucket;
        }
        if (props.crossRegionReplicationBuckets) {
            const pipelineRegion = this.requireRegion();
            return props.crossRegionReplicationBuckets[pipelineRegion];
        }
        return undefined;
    }
    calculateInsertIndexFromPlacement(placement) {
        // check if at most one placement property was provided
        const providedPlacementProps = ['rightBefore', 'justAfter', 'atIndex']
            .filter((prop) => placement[prop] !== undefined);
        if (providedPlacementProps.length > 1) {
            throw new Error('Error adding Stage to the Pipeline: ' +
                'you can only provide at most one placement property, but ' +
                `'${providedPlacementProps.join(', ')}' were given`);
        }
        if (placement.rightBefore !== undefined) {
            const targetIndex = this.findStageIndex(placement.rightBefore);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it before, '${placement.rightBefore.stageName}', was not found`);
            }
            return targetIndex;
        }
        if (placement.justAfter !== undefined) {
            const targetIndex = this.findStageIndex(placement.justAfter);
            if (targetIndex === -1) {
                throw new Error('Error adding Stage to the Pipeline: ' +
                    `the requested Stage to add it after, '${placement.justAfter.stageName}', was not found`);
            }
            return targetIndex + 1;
        }
        return this.stageCount;
    }
    findStageIndex(targetStage) {
        return this._stages.findIndex(stage => stage === targetStage);
    }
    validateSourceActionLocations() {
        const errors = new Array();
        let firstStage = true;
        for (const stage of this._stages) {
            const onlySourceActionsPermitted = firstStage;
            for (const action of stage.actionDescriptors) {
                errors.push(...validation_1.validateSourceAction(onlySourceActionsPermitted, action.category, action.actionName, stage.stageName));
            }
            firstStage = false;
        }
        return errors;
    }
    validateHasStages() {
        if (this.stageCount < 2) {
            return ['Pipeline must have at least two stages'];
        }
        return [];
    }
    validateStages() {
        const ret = new Array();
        for (const stage of this._stages) {
            ret.push(...stage.validate());
        }
        return ret;
    }
    validateArtifacts() {
        const ret = new Array();
        const producers = {};
        const firstConsumers = {};
        for (const [stageIndex, stage] of enumerate(this._stages)) {
            // For every output artifact, get the producer
            for (const action of stage.actionDescriptors) {
                const actionLoc = new PipelineLocation(stageIndex, stage, action);
                for (const outputArtifact of action.outputs) {
                    // output Artifacts always have a name set
                    const name = outputArtifact.artifactName;
                    if (producers[name]) {
                        ret.push(`Both Actions '${producers[name].actionName}' and '${action.actionName}' are producting Artifact '${name}'. Every artifact can only be produced once.`);
                        continue;
                    }
                    producers[name] = actionLoc;
                }
                // For every input artifact, get the first consumer
                for (const inputArtifact of action.inputs) {
                    const name = inputArtifact.artifactName;
                    if (!name) {
                        ret.push(`Action '${action.actionName}' is using an unnamed input Artifact, which is not being produced in this pipeline`);
                        continue;
                    }
                    firstConsumers[name] = firstConsumers[name] ? firstConsumers[name].first(actionLoc) : actionLoc;
                }
            }
        }
        // Now validate that every input artifact is produced before it's
        // being consumed.
        for (const [artifactName, consumerLoc] of Object.entries(firstConsumers)) {
            const producerLoc = producers[artifactName];
            if (!producerLoc) {
                ret.push(`Action '${consumerLoc.actionName}' is using input Artifact '${artifactName}', which is not being produced in this pipeline`);
                continue;
            }
            if (consumerLoc.beforeOrEqual(producerLoc)) {
                ret.push(`${consumerLoc} is consuming input Artifact '${artifactName}' before it is being produced at ${producerLoc}`);
            }
        }
        return ret;
    }
    renderArtifactStoresProperty() {
        if (!this.crossRegion) {
            return undefined;
        }
        // add the Pipeline's artifact store
        const primaryRegion = this.requireRegion();
        this._crossRegionSupport[primaryRegion] = {
            replicationBucket: this.artifactBucket,
            stack: core_1.Stack.of(this),
        };
        return Object.entries(this._crossRegionSupport).map(([region, support]) => ({
            region,
            artifactStore: this.renderArtifactStore(support.replicationBucket),
        }));
    }
    renderArtifactStoreProperty() {
        if (this.crossRegion) {
            return undefined;
        }
        return this.renderPrimaryArtifactStore();
    }
    renderPrimaryArtifactStore() {
        return this.renderArtifactStore(this.artifactBucket);
    }
    renderArtifactStore(bucket) {
        let encryptionKey;
        const bucketKey = bucket.encryptionKey;
        if (bucketKey) {
            encryptionKey = {
                type: 'KMS',
                id: bucketKey.keyArn,
            };
        }
        return {
            type: 'S3',
            location: bucket.bucketName,
            encryptionKey,
        };
    }
    get crossRegion() {
        if (this.crossRegionBucketsPassed) {
            return true;
        }
        return this._stages.some(stage => stage.actionDescriptors.some(action => action.region !== undefined));
    }
    renderStages() {
        return this._stages.map(stage => stage.render());
    }
    requireRegion() {
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Pipeline stack which uses cross-environment actions must have an explicitly set region');
        }
        return region;
    }
    requireApp() {
        const app = this.node.root;
        if (!app || !core_1.App.isApp(app)) {
            throw new Error('Pipeline stack which uses cross-environment actions must be part of a CDK app');
        }
        return app;
    }
}
exports.Pipeline = Pipeline;
function enumerate(xs) {
    const ret = new Array();
    for (let i = 0; i < xs.length; i++) {
        ret.push([i, xs[i]]);
    }
    return ret;
}
class PipelineLocation {
    constructor(stageIndex, stage, action) {
        this.stageIndex = stageIndex;
        this.stage = stage;
        this.action = action;
    }
    get stageName() {
        return this.stage.stageName;
    }
    get actionName() {
        return this.action.actionName;
    }
    /**
     * Returns whether a is before or the same order as b
     */
    beforeOrEqual(rhs) {
        if (this.stageIndex !== rhs.stageIndex) {
            return rhs.stageIndex < rhs.stageIndex;
        }
        return this.action.runOrder <= rhs.action.runOrder;
    }
    /**
     * Returns the first location between this and the other one
     */
    first(rhs) {
        return this.beforeOrEqual(rhs) ? this : rhs;
    }
    toString() {
        // runOrders are 1-based, so make the stageIndex also 1-based otherwise it's going to be confusing.
        return `Stage ${this.stageIndex + 1} Action ${this.action.runOrder} ('${this.stageName}'/'${this.actionName}')`;
    }
}
//# sourceMappingURL=data:application/json;base64,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