"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
// Imports
const logs = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    deployment.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W45',
                    reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
                }]
        }
    };
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = new logs.LogGroup(scope, 'ApiAccessLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    if (params.awsIntegrationProps) {
        const overridenProps = utils_1.overrideProps(baseProps, params.awsIntegrationProps);
        apiGatewayIntegration = new api.AwsIntegration(overridenProps);
    }
    else {
        apiGatewayIntegration = new api.AwsIntegration(baseProps);
    }
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    if (params.methodOptions) {
        const overridenProps = utils_1.overrideProps(defaultMethodOptions, params.methodOptions);
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    }
    else {
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, defaultMethodOptions);
    }
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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