"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontDistributionForS3 = exports.CloudFrontDistributionForApiGateway = void 0;
const cloudfront = require("@aws-cdk/aws-cloudfront");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cloudfront_distribution_defaults_1 = require("./cloudfront-distribution-defaults");
const utils_1 = require("./utils");
const lambda_helper_1 = require("./lambda-helper");
const s3_bucket_helper_1 = require("./s3-bucket-helper");
// Override Cfn_Nag rule: Cloudfront TLS-1.2 rule (https://github.com/stelligent/cfn_nag/issues/384)
function updateSecurityPolicy(cfDistribution) {
    const cfnCfDistribution = cfDistribution.node.defaultChild;
    cfnCfDistribution.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W70',
                    reason: `Since the distribution uses the CloudFront domain name, CloudFront automatically sets the security policy to TLSv1 regardless of the value of MinimumProtocolVersion`
                }]
        }
    };
    return cfDistribution;
}
// Lambda@Edge function to insert the HTTP Security Headers into the response coming from the origin servers
// and before it is sent to the client
function defaultLambdaEdgeFunction(scope) {
    const edgeLambdaFunc = lambda_helper_1.deployLambdaFunction(scope, {
        code: new lambda.InlineCode("exports.handler = (event, context, callback) => { \
          const response = event.Records[0].cf.response; \
          const headers = response.headers; \
          headers['x-xss-protection'] = [ \
            { \
              key: 'X-XSS-Protection', \
              value: '1; mode=block' \
            } \
          ]; \
          headers['x-frame-options'] = [ \
            { \
              key: 'X-Frame-Options', \
              value: 'DENY' \
            } \
          ]; \
          headers['x-content-type-options'] = [ \
            { \
              key: 'X-Content-Type-Options', \
              value: 'nosniff' \
            } \
          ]; \
          headers['strict-transport-security'] = [ \
            { \
              key: 'Strict-Transport-Security', \
              value: 'max-age=63072000; includeSubdomains; preload' \
            } \
          ]; \
          headers['referrer-policy'] = [ \
            { \
              key: 'Referrer-Policy', \
              value: 'same-origin' \
            } \
          ]; \
          headers['content-security-policy'] = [ \
            { \
              key: 'Content-Security-Policy', \
              value: \"default-src 'none'; base-uri 'self'; img-src 'self'; script-src 'self'; style-src 'self' https:; object-src 'none'; frame-ancestors 'none'; font-src 'self' https:; form-action 'self'; manifest-src 'self'; connect-src 'self'\" \
             } \
          ]; \
          callback(null, response); \
        };"),
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler'
    }, 'SetHttpSecurityHeaders');
    return edgeLambdaFunc;
}
function CloudFrontDistributionForApiGateway(scope, apiEndPoint, cloudFrontDistributionProps, httpSecurityHeaders) {
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    let defaultprops;
    let edgeLambdaVersion;
    let loggingBucket;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.loggingConfig) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, cloudFrontDistributionProps.loggingConfig.bucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForApiGatewayProps(apiEndPoint, loggingBucket, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    return [cfDistribution, edgeLambdaVersion, loggingBucket];
}
exports.CloudFrontDistributionForApiGateway = CloudFrontDistributionForApiGateway;
function CloudFrontDistributionForS3(scope, sourceBucket, cloudFrontDistributionProps, httpSecurityHeaders) {
    // Create CloudFront Origin Access Identity User
    const cfnOrigAccessId = new cloudfront.CfnCloudFrontOriginAccessIdentity(scope, 'CloudFrontOriginAccessIdentity', {
        cloudFrontOriginAccessIdentityConfig: {
            comment: 'Access S3 bucket content only through CloudFront'
        }
    });
    const oaiImported = cloudfront.OriginAccessIdentity.fromOriginAccessIdentityName(scope, 'OAIImported', cfnOrigAccessId.ref);
    let defaultprops;
    let edgeLambdaVersion;
    let loggingBucket;
    const _httpSecurityHeaders = (httpSecurityHeaders !== undefined && httpSecurityHeaders === false) ? false : true;
    if (_httpSecurityHeaders) {
        edgeLambdaVersion = new lambda.Version(scope, "SetHttpSecurityHeadersVersion", {
            lambda: defaultLambdaEdgeFunction(scope)
        });
    }
    if (cloudFrontDistributionProps && cloudFrontDistributionProps.loggingConfig) {
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, cloudFrontDistributionProps.loggingConfig.bucket, oaiImported, _httpSecurityHeaders, edgeLambdaVersion);
    }
    else {
        loggingBucket = s3_bucket_helper_1.createLoggingBucket(scope, 'CloudfrontLoggingBucket');
        defaultprops = cloudfront_distribution_defaults_1.DefaultCloudFrontWebDistributionForS3Props(sourceBucket, loggingBucket, oaiImported, _httpSecurityHeaders, edgeLambdaVersion);
    }
    const cfprops = cloudFrontDistributionProps ? utils_1.overrideProps(defaultprops, cloudFrontDistributionProps) : defaultprops;
    // Create the Cloudfront Distribution
    const cfDistribution = new cloudfront.CloudFrontWebDistribution(scope, 'CloudFrontDistribution', cfprops);
    updateSecurityPolicy(cfDistribution);
    // Add S3 Bucket Policy to allow s3:GetObject for CloudFront Origin Access Identity User
    sourceBucket.addToResourcePolicy(new iam.PolicyStatement({
        actions: ['s3:GetObject'],
        resources: [sourceBucket.arnForObjects('*')],
        principals: [new iam.CanonicalUserPrincipal(cfnOrigAccessId.attrS3CanonicalUserId)]
    }));
    // Extract the CfnBucketPolicy from the sourceBucket
    const bucketPolicy = sourceBucket.policy;
    const sourceBucketPolicy = bucketPolicy.node.findChild('Resource');
    sourceBucketPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'F16',
                    reason: `Public website bucket policy requires a wildcard principal`
                }]
        }
    };
    return [cfDistribution, edgeLambdaVersion, loggingBucket];
}
exports.CloudFrontDistributionForS3 = CloudFrontDistributionForS3;
//# sourceMappingURL=data:application/json;base64,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