"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildKinesisStreamCWAlarms = exports.buildKinesisStream = void 0;
// Imports
const kinesis = require("@aws-cdk/aws-kinesis");
const kinesis_streams_defaults_1 = require("./kinesis-streams-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildKinesisStream(scope, props) {
    if (props.existingStreamObj) {
        return props.existingStreamObj;
    }
    // Setup the stream properties
    let kinesisStreamProps;
    if (props.kinesisStreamProps) {
        // If property overrides have been provided, incorporate them and deploy
        kinesisStreamProps = utils_1.overrideProps(kinesis_streams_defaults_1.DefaultStreamProps, props.kinesisStreamProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        kinesisStreamProps = kinesis_streams_defaults_1.DefaultStreamProps;
    }
    // Create the stream and return
    return new kinesis.Stream(scope, 'KinesisStream', kinesisStreamProps);
}
exports.buildKinesisStream = buildKinesisStream;
function buildKinesisStreamCWAlarms(scope) {
    // Setup CW Alarms for KinesisStream
    const alarms = new Array();
    // Alarm if Max (GetRecords.IteratorAgeMilliseconds): >= 12 hours (50% of 24 hours default retention period)
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamGetRecordsIteratorAgeAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'GetRecords.IteratorAgeMilliseconds'
        }),
        threshold: 2592000,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.minutes(5),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Consumer Record Processing Falling Behind, there is risk for data loss due to record expiration.'
    }));
    // Alarm if Avg (ReadProvisionedThroughputExceeded): > 0
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamReadProvisionedThroughputExceededAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'ReadProvisionedThroughputExceeded'
        }),
        threshold: 0,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.minutes(5),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
        alarmDescription: 'Consumer Application is Reading at a Slower Rate Than Expected.'
    }));
    return alarms;
}
exports.buildKinesisStreamCWAlarms = buildKinesisStreamCWAlarms;
//# sourceMappingURL=data:application/json;base64,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