"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.deployLambdaFunction = exports.buildLambdaFunction = void 0;
const lambda = require("@aws-cdk/aws-lambda");
const iam = require("@aws-cdk/aws-iam");
const lambda_defaults_1 = require("./lambda-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
function buildLambdaFunction(scope, props) {
    // Conditional lambda function creation
    if (!props.existingLambdaObj) {
        if (props.lambdaFunctionProps) {
            return deployLambdaFunction(scope, props.lambdaFunctionProps);
        }
        else {
            throw Error('Either existingLambdaObj or lambdaFunctionProps is required');
        }
    }
    else {
        return props.existingLambdaObj;
    }
}
exports.buildLambdaFunction = buildLambdaFunction;
function deployLambdaFunction(scope, lambdaFunctionProps, functionId) {
    var _a, _b;
    const _functionId = functionId ? functionId : 'LambdaFunction';
    const _functionRoleId = _functionId + 'ServiceRole';
    // Setup the IAM Role for Lambda Service
    const lambdaServiceRole = new iam.Role(scope, _functionRoleId, {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        inlinePolicies: {
            LambdaFunctionServiceRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:PutLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:/aws/lambda/*`]
                    })]
            })
        }
    });
    // Override the DefaultFunctionProps with user provided  lambdaFunctionProps
    const _lambdaFunctionProps = utils_1.overrideProps(lambda_defaults_1.DefaultLambdaFunctionProps(lambdaServiceRole), lambdaFunctionProps);
    const lambdafunction = new lambda.Function(scope, _functionId, _lambdaFunctionProps);
    if (lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_10_X ||
        lambdaFunctionProps.runtime === lambda.Runtime.NODEJS_12_X) {
        lambdafunction.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
    }
    const cfnLambdafunction = lambdafunction.node.findChild('Resource');
    cfnLambdafunction.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W58',
                    reason: `Lambda functions has the required permission to write CloudWatch Logs. It uses custom policy instead of arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole with more tighter permissions.`
                }]
        }
    };
    // Find the X-Ray IAM Policy
    const cfnLambdafunctionDefPolicy = (_b = (_a = lambdafunction.role) === null || _a === void 0 ? void 0 : _a.node.tryFindChild('DefaultPolicy')) === null || _b === void 0 ? void 0 : _b.node.findChild('Resource');
    // Add the CFN NAG suppress to allow for "Resource": "*" for AWS X-Ray
    cfnLambdafunctionDefPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `Lambda needs the following minimum required permissions to send trace data to X-Ray.`
                }]
        }
    };
    return lambdafunction;
}
exports.deployLambdaFunction = deployLambdaFunction;
//# sourceMappingURL=data:application/json;base64,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