"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.flagOverriddenDefaults = void 0;
const deepdiff = require("deep-diff");
const utils_1 = require("./utils");
/**
 * Emits a warning to the console when a prescriptive default value is overridden by the user.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 */
function flagOverriddenDefaults(defaultProps, userProps) {
    // Compare the properties and return any overrides
    const overrides = findOverrides(defaultProps, userProps);
    // Emit a warning for each override
    for (let i = 0; i < ((overrides !== undefined) ? overrides.length : 0); i++) {
        const e = Object.assign(overrides[i]);
        // Determine appropriate logging granularity
        const valuesAreReadable = (checkReadability(e.lhs) &&
            checkReadability(e.rhs));
        // Format the path for readability
        const path = formatOverridePath(e.path);
        // Output
        const details = (valuesAreReadable) ? ` Default value: '${e.lhs}'. You provided: '${e.rhs}'.` : '';
        utils_1.printWarning(`An override has been provided for the property: ${path}.` + details);
    }
}
exports.flagOverriddenDefaults = flagOverriddenDefaults;
/** The prefilter function returns true for any filtered path/key that should be excluded from the diff check.
 * Any Construct Props using cdk.Duration type is not properly handled by
 * 'deep-diff' library, whenever it encounters a Duration object, it throws the exception
 * 'Argument to Intrinsic must be a plain value object', so such props are excluded from the diff check.
 */
function _prefilter(_path, _key) {
    const prefilters = ['maxRecordAge', 'expiration', 'transitionAfter'];
    if (prefilters.indexOf(_key) >= 0) {
        return true;
    }
    return false;
}
/**
 * Performs a diff check of the userProps against the defaultProps to detect overridden properties.
 * @param {object} defaultProps the prescriptive defaults for the pattern.
 * @param {object} userProps the properties provided by the user, to be compared against the defaultProps.
 * @return {Array} an array containing the overridden values.
 */
function findOverrides(defaultProps, userProps) {
    const diff = deepdiff.diff(defaultProps, userProps, _prefilter);
    // Filter the results
    return (diff !== undefined) ? diff === null || diff === void 0 ? void 0 : diff.filter((e) => {
        var _a, _b;
        return (e.kind === 'E' && // only return overrides
            !((_a = e.path) === null || _a === void 0 ? void 0 : _a.includes('node')) && // stop traversing once the object graph hits the node
            !((_b = e.path) === null || _b === void 0 ? void 0 : _b.includes('bind')) // stop traversing once the object graph hits internal functions
        );
    }) : [];
}
/**
 * Converts the path object from the deep-diff module to a user-readable, bracket notation format.
 * @param {string | string[]} path either a string value or an array of strings.
 * @return {string} the formatted override path.
 */
function formatOverridePath(path) {
    return (path !== undefined && path.length > 1) ? path.toString()
        .replace(/,/g, '][')
        .replace(/\]{1}/, '')
        .replace(/$/, ']') : path;
}
/**
 * Check the readability of an input value and, in the context of the override warning service, return true if it
 * meets the established criteria. This function is used to determine whether more-detailed log output can be given.
 * @param {any} value input to be evaluated against the given criteria.
 * @return {boolean} true if the value meets the given criteria.
 * @return {boolean} false if the value does not meet the given criteria.
 */
function checkReadability(value) {
    return (typeof (value) === 'string' && // strings only
        !value.includes('${Token[') && // no circular refs
        value !== '' // no empty strings
    );
}
//# sourceMappingURL=data:application/json;base64,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