"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createLoggingBucket = exports.applySecureBucketPolicy = exports.buildS3Bucket = void 0;
const s3 = require("@aws-cdk/aws-s3");
const s3_bucket_defaults_1 = require("./s3-bucket-defaults");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const rule_1 = require("@aws-cdk/aws-s3/lib/rule");
const duration_1 = require("@aws-cdk/core/lib/duration");
function buildS3Bucket(scope, props, bucketId) {
    if (props.bucketProps) {
        return s3BucketWithLogging(scope, props.bucketProps, bucketId);
    }
    else {
        return s3BucketWithLogging(scope, s3_bucket_defaults_1.DefaultS3Props(), bucketId);
    }
}
exports.buildS3Bucket = buildS3Bucket;
function applySecureBucketPolicy(s3Bucket) {
    // Apply bucket policy to enforce encryption of data in transit
    s3Bucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${s3Bucket.bucketArn}/*`
        ],
        actions: ['*'],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
exports.applySecureBucketPolicy = applySecureBucketPolicy;
function createLoggingBucket(scope, bucketId) {
    // Create the Logging Bucket
    const loggingBucket = new s3.Bucket(scope, bucketId, s3_bucket_defaults_1.DefaultS3Props());
    applySecureBucketPolicy(loggingBucket);
    // Extract the CfnBucket from the loggingBucket
    const loggingBucketResource = loggingBucket.node.findChild('Resource');
    // Override accessControl configuration and add metadata for the logging bucket
    loggingBucketResource.addPropertyOverride('AccessControl', 'LogDeliveryWrite');
    // Turn off Versioning for the logging bucket as objects will be written only ONCE
    loggingBucketResource.addPropertyDeletionOverride('VersioningConfiguration.Status');
    // Remove the default LifecycleConfiguration for the Logging Bucket
    loggingBucketResource.addPropertyDeletionOverride('LifecycleConfiguration.Rules');
    let _reason = "This S3 bucket is used as the access logging bucket for another bucket";
    if (bucketId === 'CloudfrontLoggingBucket') {
        _reason = "This S3 bucket is used as the access logging bucket for CloudFront Distribution";
    }
    loggingBucketResource.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W35',
                    reason: _reason
                }]
        }
    };
    return loggingBucket;
}
exports.createLoggingBucket = createLoggingBucket;
function s3BucketWithLogging(scope, s3BucketProps, bucketId) {
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: rule_1.StorageClass.GLACIER,
                    transitionAfter: duration_1.Duration.days(90)
                }]
        }];
    // Create the Application Bucket
    let bucketprops;
    let loggingBucket;
    const _bucketId = bucketId ? bucketId + 'S3Bucket' : 'S3Bucket';
    const _loggingBucketId = bucketId ? bucketId + 'S3LoggingBucket' : 'S3LoggingBucket';
    if (s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.serverAccessLogsBucket) {
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if (s3BucketProps.versioned === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(undefined, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props();
        }
    }
    else {
        // Create the Logging Bucket
        loggingBucket = createLoggingBucket(scope, _loggingBucketId);
        // Attach the Default Life Cycle policy ONLY IF the versioning is ENABLED
        if ((s3BucketProps === null || s3BucketProps === void 0 ? void 0 : s3BucketProps.versioned) === undefined || s3BucketProps.versioned) {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket, lifecycleRules);
        }
        else {
            bucketprops = s3_bucket_defaults_1.DefaultS3Props(loggingBucket);
        }
    }
    if (s3BucketProps) {
        bucketprops = utils_1.overrideProps(bucketprops, s3BucketProps);
    }
    const s3Bucket = new s3.Bucket(scope, _bucketId, bucketprops);
    applySecureBucketPolicy(s3Bucket);
    return [s3Bucket, loggingBucket];
}
//# sourceMappingURL=data:application/json;base64,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