"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
// Imports
const aws_logs_1 = require("@aws-cdk/aws-logs");
const cdk = require("@aws-cdk/core");
const smDefaults = require("./step-function-defaults");
const cloudwatch_log_group_defaults_1 = require("./cloudwatch-log-group-defaults");
const sfn = require("@aws-cdk/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps) {
    let logGroup;
    // Configure Cloudwatch log group for Step function State Machine
    if (!stateMachineProps.logs) {
        logGroup = new aws_logs_1.LogGroup(scope, 'StateMachineLogGroup', cloudwatch_log_group_defaults_1.DefaultLogGroupProps());
    }
    else {
        logGroup = stateMachineProps.logs.destination;
    }
    // Override the defaults with the user provided props
    const _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    cfnDefaultPolicy.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W12',
                    reason: `The 'LogDelivery' actions do not support resource-level authorizations`
                }]
        }
    };
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Sum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted(),
        threshold: 1,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.seconds(300),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,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