"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lambda = require("@aws-cdk/aws-lambda");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
function deployRegionalApiGateway(stack) {
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    return defaults.RegionalLambdaRestApi(stack, fn);
}
function setupRestApi(stack, apiProps) {
    const [restApi] = defaults.GlobalRestApi(stack, apiProps);
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    // Setup the API Gateway Integration
    const apiGatewayIntegration = new api.AwsIntegration({
        service: "sqs",
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            requestParameters: {
                "integration.request.header.Content-Type": "'application/x-www-form-urlencoded'"
            },
            requestTemplates: {
                "application/x-www-form-urlencoded": "Action=SendMessage&MessageBody=$util.urlEncode(\"$input.body\")&MessageAttribute.1.Name=queryParam1&MessageAttribute.1.Value.StringValue=$input.params(\"query_param_1\")&MessageAttribute.1.Value.DataType=String"
            },
            integrationResponses: [
                {
                    statusCode: "200",
                    responseTemplates: {
                        "text/html": "Success"
                    }
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        },
        path: '11112222' + "/" + 'thisqueuequeueName'
    });
    // Setup the API Gateway method(s)
    apiGatewayResource.addMethod('POST', apiGatewayIntegration, {
        requestParameters: {
            "method.request.querystring.query_param_1": true
        },
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ]
    });
}
test('snapshot test RegionalApiGateway default params', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        description: 'Hello World'
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            Description: "Hello World",
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "LambdaRestApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test override for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn, {
        handler: fn,
        restApiName: "HelloWorld"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "EDGE"
                ]
            },
            Name: "HelloWorld"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
test('Test ApiGateway::Account resource for RegionalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test ApiGateway::Account resource for GlobalApiGateway', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.GlobalLambdaRestApi(stack, fn);
    expect(stack).toHaveResource('AWS::ApiGateway::Account', {
        CloudWatchRoleArn: {
            "Fn::GetAtt": [
                "LambdaRestApiCloudWatchRoleF339D4E6",
                "Arn"
            ]
        }
    });
});
test('Test default RestApi deployment w/o ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test default RestApi deployment w/ ApiGatewayProps', () => {
    const stack = new core_1.Stack();
    setupRestApi(stack, {
        restApiName: "customRestApi"
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Assertion 2
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Name: "customRestApi"
    });
});
test('Test default RestApi deployment for Cloudwatch loggroup', () => {
    const stack = new core_1.Stack();
    deployRegionalApiGateway(stack);
    expect(stack).toHaveResource('AWS::Logs::LogGroup', {
        UpdateReplacePolicy: "Retain",
        DeletionPolicy: "Retain"
    }, assert_1.ResourcePart.CompleteDefinition);
    expect(stack).toHaveResource('AWS::ApiGateway::Stage', {
        AccessLogSetting: {
            DestinationArn: {
                "Fn::GetAtt": [
                    "ApiAccessLogGroupCEA70788",
                    "Arn"
                ]
            },
            Format: "{\"requestId\":\"$context.requestId\",\"ip\":\"$context.identity.sourceIp\",\"user\":\"$context.identity.user\",\"caller\":\"$context.identity.caller\",\"requestTime\":\"$context.requestTime\",\"httpMethod\":\"$context.httpMethod\",\"resourcePath\":\"$context.resourcePath\",\"status\":\"$context.status\",\"protocol\":\"$context.protocol\",\"responseLength\":\"$context.responseLength\"}",
        },
    });
});
test('Test addMethodToApiResource with action', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const getRequestTemplate = "{}";
    // Add Method
    defaults.addProxyMethodToApiResource({
        action: "Query",
        service: "dynamodb",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "GET",
        requestTemplate: getRequestTemplate
    });
    // Add Method
    defaults.addProxyMethodToApiResource({
        path: '11112222' + "/" + 'thisqueuequeueName',
        service: "sqs",
        apiResource: apiGatewayResource,
        apiGatewayRole,
        apiMethod: "PUT",
        requestTemplate: getRequestTemplate
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    // Error scenario: missing action and path
    try {
        defaults.addProxyMethodToApiResource({
            service: "sqs",
            apiResource: apiGatewayResource,
            apiGatewayRole,
            apiMethod: "DELETE",
            requestTemplate: getRequestTemplate
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('Test default RestApi w/ request model and validator', () => {
    const stack = new core_1.Stack();
    const [restApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = restApi.root.addResource('api-gateway-resource');
    const validator = restApi.addRequestValidator('default-validator', {
        requestValidatorName: 'default-validator',
        validateRequestBody: true
    });
    defaults.addProxyMethodToApiResource({
        service: "kinesis",
        action: "PutRecord",
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "{}",
        contentType: "'x-amz-json-1.1'",
        requestValidator: validator,
        requestModel: { "application/json": api.Model.EMPTY_MODEL }
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
    expect(stack).toHaveResource('AWS::ApiGateway::RequestValidator', {
        Name: "default-validator",
        ValidateRequestBody: true
    });
    expect(stack).toHaveResourceLike('AWS::ApiGateway::Method', {
        RequestModels: { "application/json": "Empty" }
    });
});
// -----------------------------------------------------------------------
// Test for Regional ApiGateway Creation
// -----------------------------------------------------------------------
test('Test for RegionalRestApiGateway', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack, {
        restApiName: "HelloWorld-RegionalApi"
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    expect(stack).toHaveResource('AWS::ApiGateway::RestApi', {
        Type: "AWS::ApiGateway::RestApi",
        Properties: {
            EndpointConfiguration: {
                Types: [
                    "REGIONAL"
                ]
            },
            Name: "HelloWorld-RegionalApi"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// -----------------------------------------------------------------------
// Tests for exception while overriding restApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding restApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    try {
        defaults.RegionalRestApi(stack, {
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Tests for exception while overriding LambdaRestApiProps using endPointTypes
// -----------------------------------------------------------------------
test('Test for Exception while overriding LambdaRestApiProps using endPointTypes', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    try {
        defaults.GlobalLambdaRestApi(stack, fn, {
            handler: fn,
            endpointTypes: [api.EndpointType.REGIONAL]
        });
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
// -----------------------------------------------------------------------
// Test for Integration Request Props Override
// -----------------------------------------------------------------------
test('Test for Integration Request Props Override', () => {
    const stack = new core_1.Stack();
    const [regionalApi] = defaults.RegionalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = regionalApi.root.addResource('hello');
    const integReqParams = { 'integration.request.path.topic-level-1': "'method.request.path.topic-level-1'" };
    const integResp = [
        {
            statusCode: "200",
            selectionPattern: "2\\d{2}",
            responseTemplates: {
                "application/json": "$input.json('$')"
            }
        }
    ];
    // Override the default Integration Request Props
    const integrationReqProps = {
        subdomain: 'abcdefgh12345',
        options: {
            requestParameters: integReqParams,
            integrationResponses: integResp,
            passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
        }
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        awsIntegrationProps: integrationReqProps
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    ResponseTemplates: {
                        "application/json": "$input.json('$')"
                    },
                    SelectionPattern: "2\\d{2}",
                    StatusCode: "200"
                }
            ],
            PassthroughBehavior: "WHEN_NO_MATCH",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
                "integration.request.path.topic-level-1": "'method.request.path.topic-level-1'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:abcdefgh12345.iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "200",
                ResponseParameters: {
                    "method.response.header.Content-Type": true
                }
            }
        ]
    });
});
// -----------------------------------------------------------------------
// Test for Method Options Override
// -----------------------------------------------------------------------
test('Test for Method Request Props Override', () => {
    const stack = new core_1.Stack();
    const [globalApi] = defaults.GlobalRestApi(stack);
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalApi.root.addResource('hello');
    const methodReqParams = { 'method.request.path.topic-level-1': true };
    const methodResp = [
        {
            statusCode: "403"
        }
    ];
    const resourceMethodOptions = {
        requestParameters: methodReqParams,
        methodResponses: methodResp,
    };
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')",
        methodOptions: resourceMethodOptions
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        HttpMethod: "POST",
        AuthorizationType: "AWS_IAM",
        Integration: {
            IntegrationHttpMethod: "POST",
            IntegrationResponses: [
                {
                    StatusCode: "200"
                },
                {
                    StatusCode: "500",
                    ResponseTemplates: {
                        "text/html": "Error"
                    },
                    SelectionPattern: "500"
                }
            ],
            PassthroughBehavior: "NEVER",
            RequestParameters: {
                "integration.request.header.Content-Type": "'application/json'",
            },
            RequestTemplates: {
                "application/json": "$input.json('$')"
            },
            Type: "AWS",
            Uri: {
                "Fn::Join": [
                    "",
                    [
                        "arn:",
                        {
                            Ref: "AWS::Partition"
                        },
                        ":apigateway:",
                        {
                            Ref: "AWS::Region"
                        },
                        `:iotdata:path/hello`
                    ]
                ]
            }
        },
        MethodResponses: [
            {
                StatusCode: "403"
            }
        ],
        RequestParameters: {
            "method.request.path.topic-level-1": true
        }
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of RestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using restApiProps', () => {
    const stack = new core_1.Stack();
    const [globalRestApi] = defaults.GlobalRestApi(stack, {
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Setup the API Gateway role
    const apiGatewayRole = new iam.Role(stack, 'api-gateway-role', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com')
    });
    // Setup the API Gateway resource
    const apiGatewayResource = globalRestApi.root.addResource('hello');
    defaults.addProxyMethodToApiResource({
        service: 'iotdata',
        path: 'hello',
        apiGatewayRole,
        apiMethod: 'POST',
        apiResource: apiGatewayResource,
        requestTemplate: "$input.json('$')"
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
// -----------------------------------------------------------------------
// Test for ApiKey Creation of LambdaRestApi
// -----------------------------------------------------------------------
test('Test for ApiKey creation using lambdaApiProps', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    };
    const fn = defaults.deployLambdaFunction(stack, lambdaFunctionProps);
    defaults.RegionalLambdaRestApi(stack, fn, {
        handler: fn,
        defaultMethodOptions: {
            apiKeyRequired: true
        }
    });
    // Assertion to check for ApiKey
    expect(stack).toHaveResourceLike("AWS::ApiGateway::Method", {
        ApiKeyRequired: true
    });
    expect(stack).toHaveResourceLike("AWS::ApiGateway::ApiKey", {
        Enabled: true
    });
    // Assertion to check for UsagePlan Api Key Mapping
    expect(stack).toHaveResourceLike("AWS::ApiGateway::UsagePlanKey", {
        KeyType: "API_KEY"
    });
});
//# sourceMappingURL=data:application/json;base64,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