"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const api = require("@aws-cdk/aws-apigateway");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("../index");
const s3 = require("@aws-cdk/aws-s3");
const cloudfront_distribution_helper_1 = require("../lib/cloudfront-distribution-helper");
require("@aws-cdk/assert/jest");
const aws_cloudfront_1 = require("@aws-cdk/aws-cloudfront");
test('cloudfront distribution for ApiGateway with default params', () => {
    const stack = new core_1.Stack();
    const lambdaFunctionProps = {
        runtime: lambda.Runtime.NODEJS_12_X,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`)
    };
    const func = new lambda.Function(stack, 'LambdaFunction', lambdaFunctionProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi', {
        handler: func
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test cloudfront for Api Gateway with user provided logging bucket', () => {
    const stack = new core_1.Stack();
    const loggingBucket = new s3.Bucket(stack, 'MyCloudfrontLoggingBucket', defaults.DefaultS3Props());
    const inProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda-test`),
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler'
    };
    const cfdProps = {
        loggingConfig: {
            bucket: loggingBucket
        }
    };
    const func = defaults.deployLambdaFunction(stack, inProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi1', {
        handler: func
    });
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api, cfdProps);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "GET",
                    "HEAD"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "MyCloudfrontLoggingBucket9AA652E8",
                        "DomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        HTTPPort: 80,
                        HTTPSPort: 443,
                        OriginKeepaliveTimeout: 5,
                        OriginProtocolPolicy: "https-only",
                        OriginReadTimeout: 30,
                        OriginSSLProtocols: [
                            "TLSv1.2"
                        ]
                    },
                    DomainName: {
                        "Fn::Select": [
                            0,
                            {
                                "Fn::Split": [
                                    "/",
                                    {
                                        "Fn::Select": [
                                            1,
                                            {
                                                "Fn::Split": [
                                                    "://",
                                                    {
                                                        "Fn::Join": [
                                                            "",
                                                            [
                                                                "https://",
                                                                {
                                                                    Ref: "RestApi1480AC499"
                                                                },
                                                                ".execute-api.",
                                                                {
                                                                    Ref: "AWS::Region"
                                                                },
                                                                ".",
                                                                {
                                                                    Ref: "AWS::URLSuffix"
                                                                },
                                                                "/",
                                                                {
                                                                    Ref: "RestApi1DeploymentStageprod4FFC9BB4"
                                                                },
                                                                "/"
                                                            ]
                                                        ]
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                ]
                            }
                        ]
                    },
                    Id: "origin1"
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
test('test cloudfront for Api Gateway override properties', () => {
    const stack = new core_1.Stack();
    const inProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda-test`),
        runtime: lambda.Runtime.PYTHON_3_6,
        handler: 'index.handler'
    };
    const func = defaults.deployLambdaFunction(stack, inProps);
    const _api = new api.LambdaRestApi(stack, 'RestApi1', {
        handler: func
    });
    const props = {
        originConfigs: [{
                customOriginSource: {
                    domainName: _api.url,
                    originProtocolPolicy: aws_cloudfront_1.OriginProtocolPolicy.HTTP_ONLY
                },
                behaviors: [{
                        isDefaultBehavior: true,
                        allowedMethods: cloudfront.CloudFrontAllowedMethods.ALL,
                        cachedMethods: cloudfront.CloudFrontAllowedCachedMethods.GET_HEAD_OPTIONS
                    }]
            }]
    };
    cloudfront_distribution_helper_1.CloudFrontDistributionForApiGateway(stack, _api, props);
    expect(stack).toHaveResourceLike("AWS::CloudFront::Distribution", {
        DistributionConfig: {
            DefaultCacheBehavior: {
                AllowedMethods: [
                    "DELETE",
                    "GET",
                    "HEAD",
                    "OPTIONS",
                    "PATCH",
                    "POST",
                    "PUT"
                ],
                CachedMethods: [
                    "GET",
                    "HEAD",
                    "OPTIONS"
                ],
                Compress: true,
                ForwardedValues: {
                    Cookies: {
                        Forward: "none"
                    },
                    QueryString: false
                },
                TargetOriginId: "origin1",
                ViewerProtocolPolicy: "redirect-to-https"
            },
            DefaultRootObject: "index.html",
            Enabled: true,
            HttpVersion: "http2",
            IPV6Enabled: true,
            Logging: {
                Bucket: {
                    "Fn::GetAtt": [
                        "CloudfrontLoggingBucket3C3EFAA7",
                        "DomainName"
                    ]
                },
                IncludeCookies: false
            },
            Origins: [
                {
                    CustomOriginConfig: {
                        HTTPPort: 80,
                        HTTPSPort: 443,
                        OriginKeepaliveTimeout: 5,
                        OriginProtocolPolicy: "http-only",
                        OriginReadTimeout: 30,
                        OriginSSLProtocols: [
                            "TLSv1.2"
                        ]
                    },
                    DomainName: {
                        "Fn::Join": [
                            "",
                            [
                                "https://",
                                {
                                    Ref: "RestApi1480AC499"
                                },
                                ".execute-api.",
                                {
                                    Ref: "AWS::Region"
                                },
                                ".",
                                {
                                    Ref: "AWS::URLSuffix"
                                },
                                "/",
                                {
                                    Ref: "RestApi1DeploymentStageprod4FFC9BB4"
                                },
                                "/"
                            ]
                        ]
                    },
                    Id: "origin1"
                }
            ],
            PriceClass: "PriceClass_100",
            ViewerCertificate: {
                CloudFrontDefaultCertificate: true
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2xvdWRmcm9udC1kaXN0cmlidXRpb24tYXBpLWdhdGV3YXktaGVscGVyLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjbG91ZGZyb250LWRpc3RyaWJ1dGlvbi1hcGktZ2F0ZXdheS1oZWxwZXIudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsNENBQTZDO0FBQzdDLHdDQUFzQztBQUN0QyxzREFBc0Q7QUFDdEQsK0NBQStDO0FBQy9DLDhDQUE4QztBQUM5QyxxQ0FBcUM7QUFDckMsc0NBQXNDO0FBQ3RDLDBGQUE0RjtBQUM1RixnQ0FBOEI7QUFDOUIsNERBQStEO0FBRS9ELElBQUksQ0FBQyw0REFBNEQsRUFBRSxHQUFHLEVBQUU7SUFDdEUsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLG1CQUFtQixHQUF5QjtRQUNoRCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1FBQ25DLE9BQU8sRUFBRSxlQUFlO1FBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO0tBQ25ELENBQUM7SUFFRixNQUFNLElBQUksR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFLG1CQUFtQixDQUFDLENBQUM7SUFDL0UsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxTQUFTLEVBQUU7UUFDbkQsT0FBTyxFQUFFLElBQUk7S0FDZCxDQUFDLENBQUM7SUFDSCxvRUFBbUMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxDQUFDLENBQUM7SUFDakQsTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxtRUFBbUUsRUFBRSxHQUFHLEVBQUU7SUFDN0UsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLGFBQWEsR0FBYyxJQUFJLEVBQUUsQ0FBQyxNQUFNLENBQUMsS0FBSyxFQUFFLDJCQUEyQixFQUFFLFFBQVEsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO0lBRTlHLE1BQU0sT0FBTyxHQUF5QjtRQUNwQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLGNBQWMsQ0FBQztRQUN2RCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1FBQ2xDLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLFFBQVEsR0FBRztRQUNmLGFBQWEsRUFBRTtZQUNiLE1BQU0sRUFBRSxhQUFhO1NBQ3RCO0tBQ0YsQ0FBQztJQUVGLE1BQU0sSUFBSSxHQUFHLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsT0FBTyxDQUFDLENBQUM7SUFFM0QsTUFBTSxJQUFJLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDcEQsT0FBTyxFQUFFLElBQUk7S0FDZCxDQUFDLENBQUM7SUFFSCxvRUFBbUMsQ0FBQyxLQUFLLEVBQUUsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO0lBQzNELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQywrQkFBK0IsRUFBRTtRQUNoRSxrQkFBa0IsRUFBRTtZQUNsQixvQkFBb0IsRUFBRTtnQkFDcEIsY0FBYyxFQUFFO29CQUNkLEtBQUs7b0JBQ0wsTUFBTTtpQkFDUDtnQkFDRCxhQUFhLEVBQUU7b0JBQ2IsS0FBSztvQkFDTCxNQUFNO2lCQUNQO2dCQUNELFFBQVEsRUFBRSxJQUFJO2dCQUNkLGVBQWUsRUFBRTtvQkFDZixPQUFPLEVBQUU7d0JBQ1AsT0FBTyxFQUFFLE1BQU07cUJBQ2hCO29CQUNELFdBQVcsRUFBRSxLQUFLO2lCQUNuQjtnQkFDRCxjQUFjLEVBQUUsU0FBUztnQkFDekIsb0JBQW9CLEVBQUUsbUJBQW1CO2FBQzFDO1lBQ0QsaUJBQWlCLEVBQUUsWUFBWTtZQUMvQixPQUFPLEVBQUUsSUFBSTtZQUNiLFdBQVcsRUFBRSxPQUFPO1lBQ3BCLFdBQVcsRUFBRSxJQUFJO1lBQ2pCLE9BQU8sRUFBRTtnQkFDUCxNQUFNLEVBQUU7b0JBQ04sWUFBWSxFQUFFO3dCQUNaLG1DQUFtQzt3QkFDbkMsWUFBWTtxQkFDYjtpQkFDRjtnQkFDRCxjQUFjLEVBQUUsS0FBSzthQUN0QjtZQUNELE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxrQkFBa0IsRUFBRTt3QkFDbEIsUUFBUSxFQUFFLEVBQUU7d0JBQ1osU0FBUyxFQUFFLEdBQUc7d0JBQ2Qsc0JBQXNCLEVBQUUsQ0FBQzt3QkFDekIsb0JBQW9CLEVBQUUsWUFBWTt3QkFDbEMsaUJBQWlCLEVBQUUsRUFBRTt3QkFDckIsa0JBQWtCLEVBQUU7NEJBQ2xCLFNBQVM7eUJBQ1Y7cUJBQ0Y7b0JBQ0QsVUFBVSxFQUFFO3dCQUNWLFlBQVksRUFBRTs0QkFDWixDQUFDOzRCQUNEO2dDQUNFLFdBQVcsRUFBRTtvQ0FDWCxHQUFHO29DQUNIO3dDQUNFLFlBQVksRUFBRTs0Q0FDWixDQUFDOzRDQUNEO2dEQUNFLFdBQVcsRUFBRTtvREFDWCxLQUFLO29EQUNMO3dEQUNFLFVBQVUsRUFBRTs0REFDVixFQUFFOzREQUNGO2dFQUNFLFVBQVU7Z0VBQ1Y7b0VBQ0UsR0FBRyxFQUFFLGtCQUFrQjtpRUFDeEI7Z0VBQ0QsZUFBZTtnRUFDZjtvRUFDRSxHQUFHLEVBQUUsYUFBYTtpRUFDbkI7Z0VBQ0QsR0FBRztnRUFDSDtvRUFDRSxHQUFHLEVBQUUsZ0JBQWdCO2lFQUN0QjtnRUFDRCxHQUFHO2dFQUNIO29FQUNFLEdBQUcsRUFBRSxxQ0FBcUM7aUVBQzNDO2dFQUNELEdBQUc7NkRBQ0o7eURBQ0Y7cURBQ0Y7aURBQ0Y7NkNBQ0Y7eUNBQ0Y7cUNBQ0Y7aUNBQ0Y7NkJBQ0Y7eUJBQ0Y7cUJBQ0Y7b0JBQ0QsRUFBRSxFQUFFLFNBQVM7aUJBQ2Q7YUFDRjtZQUNELFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsaUJBQWlCLEVBQUU7Z0JBQ2pCLDRCQUE0QixFQUFFLElBQUk7YUFDbkM7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFEQUFxRCxFQUFFLEdBQUcsRUFBRTtJQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBRTFCLE1BQU0sT0FBTyxHQUF5QjtRQUNwQyxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLGNBQWMsQ0FBQztRQUN2RCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1FBQ2xDLE9BQU8sRUFBRSxlQUFlO0tBQ3pCLENBQUM7SUFFRixNQUFNLElBQUksR0FBRyxRQUFRLENBQUMsb0JBQW9CLENBQUMsS0FBSyxFQUFFLE9BQU8sQ0FBQyxDQUFDO0lBRTNELE1BQU0sSUFBSSxHQUFHLElBQUksR0FBRyxDQUFDLGFBQWEsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQ3BELE9BQU8sRUFBRSxJQUFJO0tBQ2QsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQThDO1FBQ3ZELGFBQWEsRUFBRSxDQUFFO2dCQUNmLGtCQUFrQixFQUFFO29CQUNoQixVQUFVLEVBQUUsSUFBSSxDQUFDLEdBQUc7b0JBQ3BCLG9CQUFvQixFQUFFLHFDQUFvQixDQUFDLFNBQVM7aUJBQ3JEO2dCQUNELFNBQVMsRUFBRSxDQUFFO3dCQUNMLGlCQUFpQixFQUFFLElBQUk7d0JBQ3ZCLGNBQWMsRUFBRSxVQUFVLENBQUMsd0JBQXdCLENBQUMsR0FBRzt3QkFDdkQsYUFBYSxFQUFFLFVBQVUsQ0FBQyw4QkFBOEIsQ0FBQyxnQkFBZ0I7cUJBQzVFLENBQUU7YUFDVixDQUFFO0tBQ0osQ0FBQztJQUVGLG9FQUFtQyxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFeEQsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGtCQUFrQixDQUFDLCtCQUErQixFQUFFO1FBQ2hFLGtCQUFrQixFQUFFO1lBQ2hCLG9CQUFvQixFQUFFO2dCQUNwQixjQUFjLEVBQUU7b0JBQ2QsUUFBUTtvQkFDUixLQUFLO29CQUNMLE1BQU07b0JBQ04sU0FBUztvQkFDVCxPQUFPO29CQUNQLE1BQU07b0JBQ04sS0FBSztpQkFDTjtnQkFDRCxhQUFhLEVBQUU7b0JBQ2IsS0FBSztvQkFDTCxNQUFNO29CQUNOLFNBQVM7aUJBQ1Y7Z0JBQ0QsUUFBUSxFQUFFLElBQUk7Z0JBQ2QsZUFBZSxFQUFFO29CQUNmLE9BQU8sRUFBRTt3QkFDUCxPQUFPLEVBQUUsTUFBTTtxQkFDaEI7b0JBQ0QsV0FBVyxFQUFFLEtBQUs7aUJBQ25CO2dCQUNELGNBQWMsRUFBRSxTQUFTO2dCQUN6QixvQkFBb0IsRUFBRSxtQkFBbUI7YUFDMUM7WUFDRCxpQkFBaUIsRUFBRSxZQUFZO1lBQy9CLE9BQU8sRUFBRSxJQUFJO1lBQ2IsV0FBVyxFQUFFLE9BQU87WUFDcEIsV0FBVyxFQUFFLElBQUk7WUFDakIsT0FBTyxFQUFFO2dCQUNQLE1BQU0sRUFBRTtvQkFDTixZQUFZLEVBQUU7d0JBQ1osaUNBQWlDO3dCQUNqQyxZQUFZO3FCQUNiO2lCQUNGO2dCQUNELGNBQWMsRUFBRSxLQUFLO2FBQ3RCO1lBQ0QsT0FBTyxFQUFFO2dCQUNQO29CQUNFLGtCQUFrQixFQUFFO3dCQUNsQixRQUFRLEVBQUUsRUFBRTt3QkFDWixTQUFTLEVBQUUsR0FBRzt3QkFDZCxzQkFBc0IsRUFBRSxDQUFDO3dCQUN6QixvQkFBb0IsRUFBRSxXQUFXO3dCQUNqQyxpQkFBaUIsRUFBRSxFQUFFO3dCQUNyQixrQkFBa0IsRUFBRTs0QkFDbEIsU0FBUzt5QkFDVjtxQkFDRjtvQkFDRCxVQUFVLEVBQUU7d0JBQ1YsVUFBVSxFQUFFOzRCQUNWLEVBQUU7NEJBQ0Y7Z0NBQ0UsVUFBVTtnQ0FDVjtvQ0FDRSxHQUFHLEVBQUUsa0JBQWtCO2lDQUN4QjtnQ0FDRCxlQUFlO2dDQUNmO29DQUNFLEdBQUcsRUFBRSxhQUFhO2lDQUNuQjtnQ0FDRCxHQUFHO2dDQUNIO29DQUNFLEdBQUcsRUFBRSxnQkFBZ0I7aUNBQ3RCO2dDQUNELEdBQUc7Z0NBQ0g7b0NBQ0UsR0FBRyxFQUFFLHFDQUFxQztpQ0FDM0M7Z0NBQ0QsR0FBRzs2QkFDSjt5QkFDRjtxQkFDRjtvQkFDRCxFQUFFLEVBQUUsU0FBUztpQkFDZDthQUNGO1lBQ0QsVUFBVSxFQUFFLGdCQUFnQjtZQUM1QixpQkFBaUIsRUFBRTtnQkFDakIsNEJBQTRCLEVBQUUsSUFBSTthQUNuQztTQUNGO0tBQ0osQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IFN5bnRoVXRpbHMgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCAqIGFzIGNsb3VkZnJvbnQgZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3VkZnJvbnQnO1xuaW1wb3J0ICogYXMgYXBpIGZyb20gJ0Bhd3MtY2RrL2F3cy1hcGlnYXRld2F5JztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCAqIGFzIGRlZmF1bHRzIGZyb20gJy4uL2luZGV4JztcbmltcG9ydCAqIGFzIHMzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMyc7XG5pbXBvcnQgeyBDbG91ZEZyb250RGlzdHJpYnV0aW9uRm9yQXBpR2F0ZXdheSB9IGZyb20gJy4uL2xpYi9jbG91ZGZyb250LWRpc3RyaWJ1dGlvbi1oZWxwZXInO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgeyBPcmlnaW5Qcm90b2NvbFBvbGljeSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZyb250JztcblxudGVzdCgnY2xvdWRmcm9udCBkaXN0cmlidXRpb24gZm9yIEFwaUdhdGV3YXkgd2l0aCBkZWZhdWx0IHBhcmFtcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcblxuICBjb25zdCBsYW1iZGFGdW5jdGlvblByb3BzOiBsYW1iZGEuRnVuY3Rpb25Qcm9wcyA9IHtcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYClcbiAgfTtcblxuICBjb25zdCBmdW5jID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ0xhbWJkYUZ1bmN0aW9uJywgbGFtYmRhRnVuY3Rpb25Qcm9wcyk7XG4gIGNvbnN0IF9hcGkgPSBuZXcgYXBpLkxhbWJkYVJlc3RBcGkoc3RhY2ssICdSZXN0QXBpJywge1xuICAgIGhhbmRsZXI6IGZ1bmNcbiAgfSk7XG4gIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JBcGlHYXRld2F5KHN0YWNrLCBfYXBpKTtcbiAgZXhwZWN0KFN5bnRoVXRpbHMudG9DbG91ZEZvcm1hdGlvbihzdGFjaykpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG5cbnRlc3QoJ3Rlc3QgY2xvdWRmcm9udCBmb3IgQXBpIEdhdGV3YXkgd2l0aCB1c2VyIHByb3ZpZGVkIGxvZ2dpbmcgYnVja2V0JywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IGxvZ2dpbmdCdWNrZXQ6IHMzLkJ1Y2tldCA9IG5ldyBzMy5CdWNrZXQoc3RhY2ssICdNeUNsb3VkZnJvbnRMb2dnaW5nQnVja2V0JywgZGVmYXVsdHMuRGVmYXVsdFMzUHJvcHMoKSk7XG5cbiAgY29uc3QgaW5Qcm9wczogbGFtYmRhLkZ1bmN0aW9uUHJvcHMgPSB7XG4gICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhLXRlc3RgKSxcbiAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM182LFxuICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICB9O1xuXG4gIGNvbnN0IGNmZFByb3BzID0ge1xuICAgIGxvZ2dpbmdDb25maWc6IHtcbiAgICAgIGJ1Y2tldDogbG9nZ2luZ0J1Y2tldFxuICAgIH1cbiAgfTtcblxuICBjb25zdCBmdW5jID0gZGVmYXVsdHMuZGVwbG95TGFtYmRhRnVuY3Rpb24oc3RhY2ssIGluUHJvcHMpO1xuXG4gIGNvbnN0IF9hcGkgPSBuZXcgYXBpLkxhbWJkYVJlc3RBcGkoc3RhY2ssICdSZXN0QXBpMScsIHtcbiAgICBoYW5kbGVyOiBmdW5jXG4gIH0pO1xuXG4gIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JBcGlHYXRld2F5KHN0YWNrLCBfYXBpLCBjZmRQcm9wcyk7XG4gIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2VMaWtlKFwiQVdTOjpDbG91ZEZyb250OjpEaXN0cmlidXRpb25cIiwge1xuICAgIERpc3RyaWJ1dGlvbkNvbmZpZzoge1xuICAgICAgRGVmYXVsdENhY2hlQmVoYXZpb3I6IHtcbiAgICAgICAgQWxsb3dlZE1ldGhvZHM6IFtcbiAgICAgICAgICBcIkdFVFwiLFxuICAgICAgICAgIFwiSEVBRFwiXG4gICAgICAgIF0sXG4gICAgICAgIENhY2hlZE1ldGhvZHM6IFtcbiAgICAgICAgICBcIkdFVFwiLFxuICAgICAgICAgIFwiSEVBRFwiXG4gICAgICAgIF0sXG4gICAgICAgIENvbXByZXNzOiB0cnVlLFxuICAgICAgICBGb3J3YXJkZWRWYWx1ZXM6IHtcbiAgICAgICAgICBDb29raWVzOiB7XG4gICAgICAgICAgICBGb3J3YXJkOiBcIm5vbmVcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAgUXVlcnlTdHJpbmc6IGZhbHNlXG4gICAgICAgIH0sXG4gICAgICAgIFRhcmdldE9yaWdpbklkOiBcIm9yaWdpbjFcIixcbiAgICAgICAgVmlld2VyUHJvdG9jb2xQb2xpY3k6IFwicmVkaXJlY3QtdG8taHR0cHNcIlxuICAgICAgfSxcbiAgICAgIERlZmF1bHRSb290T2JqZWN0OiBcImluZGV4Lmh0bWxcIixcbiAgICAgIEVuYWJsZWQ6IHRydWUsXG4gICAgICBIdHRwVmVyc2lvbjogXCJodHRwMlwiLFxuICAgICAgSVBWNkVuYWJsZWQ6IHRydWUsXG4gICAgICBMb2dnaW5nOiB7XG4gICAgICAgIEJ1Y2tldDoge1xuICAgICAgICAgIFwiRm46OkdldEF0dFwiOiBbXG4gICAgICAgICAgICBcIk15Q2xvdWRmcm9udExvZ2dpbmdCdWNrZXQ5QUE2NTJFOFwiLFxuICAgICAgICAgICAgXCJEb21haW5OYW1lXCJcbiAgICAgICAgICBdXG4gICAgICAgIH0sXG4gICAgICAgIEluY2x1ZGVDb29raWVzOiBmYWxzZVxuICAgICAgfSxcbiAgICAgIE9yaWdpbnM6IFtcbiAgICAgICAge1xuICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgSFRUUFBvcnQ6IDgwLFxuICAgICAgICAgICAgSFRUUFNQb3J0OiA0NDMsXG4gICAgICAgICAgICBPcmlnaW5LZWVwYWxpdmVUaW1lb3V0OiA1LFxuICAgICAgICAgICAgT3JpZ2luUHJvdG9jb2xQb2xpY3k6IFwiaHR0cHMtb25seVwiLFxuICAgICAgICAgICAgT3JpZ2luUmVhZFRpbWVvdXQ6IDMwLFxuICAgICAgICAgICAgT3JpZ2luU1NMUHJvdG9jb2xzOiBbXG4gICAgICAgICAgICAgIFwiVExTdjEuMlwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBEb21haW5OYW1lOiB7XG4gICAgICAgICAgICBcIkZuOjpTZWxlY3RcIjogW1xuICAgICAgICAgICAgICAwLFxuICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgXCJGbjo6U3BsaXRcIjogW1xuICAgICAgICAgICAgICAgICAgXCIvXCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFwiRm46OlNlbGVjdFwiOiBbXG4gICAgICAgICAgICAgICAgICAgICAgMSxcbiAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICBcIkZuOjpTcGxpdFwiOiBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgIFwiOi8vXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwiXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICBbXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwiaHR0cHM6Ly9cIixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJSZXN0QXBpMTQ4MEFDNDk5XCJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXCIuZXhlY3V0ZS1hcGkuXCIsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcIi5cIixcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFwiL1wiLFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgUmVmOiBcIlJlc3RBcGkxRGVwbG95bWVudFN0YWdlcHJvZDRGRkM5QkI0XCJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXCIvXCJcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgICAgIF1cbiAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBdXG4gICAgICAgICAgICAgIH1cbiAgICAgICAgICAgIF1cbiAgICAgICAgICB9LFxuICAgICAgICAgIElkOiBcIm9yaWdpbjFcIlxuICAgICAgICB9XG4gICAgICBdLFxuICAgICAgUHJpY2VDbGFzczogXCJQcmljZUNsYXNzXzEwMFwiLFxuICAgICAgVmlld2VyQ2VydGlmaWNhdGU6IHtcbiAgICAgICAgQ2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZTogdHJ1ZVxuICAgICAgfVxuICAgIH1cbiAgfSk7XG59KTtcblxudGVzdCgndGVzdCBjbG91ZGZyb250IGZvciBBcGkgR2F0ZXdheSBvdmVycmlkZSBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IGluUHJvcHM6IGxhbWJkYS5GdW5jdGlvblByb3BzID0ge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYS10ZXN0YCksXG4gICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfNixcbiAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgfTtcblxuICBjb25zdCBmdW5jID0gZGVmYXVsdHMuZGVwbG95TGFtYmRhRnVuY3Rpb24oc3RhY2ssIGluUHJvcHMpO1xuXG4gIGNvbnN0IF9hcGkgPSBuZXcgYXBpLkxhbWJkYVJlc3RBcGkoc3RhY2ssICdSZXN0QXBpMScsIHtcbiAgICBoYW5kbGVyOiBmdW5jXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBjbG91ZGZyb250LkNsb3VkRnJvbnRXZWJEaXN0cmlidXRpb25Qcm9wcyA9IHtcbiAgICBvcmlnaW5Db25maWdzOiBbIHtcbiAgICAgIGN1c3RvbU9yaWdpblNvdXJjZToge1xuICAgICAgICAgIGRvbWFpbk5hbWU6IF9hcGkudXJsLFxuICAgICAgICAgIG9yaWdpblByb3RvY29sUG9saWN5OiBPcmlnaW5Qcm90b2NvbFBvbGljeS5IVFRQX09OTFlcbiAgICAgICAgfSxcbiAgICAgICAgYmVoYXZpb3JzOiBbIHtcbiAgICAgICAgICAgICAgICBpc0RlZmF1bHRCZWhhdmlvcjogdHJ1ZSxcbiAgICAgICAgICAgICAgICBhbGxvd2VkTWV0aG9kczogY2xvdWRmcm9udC5DbG91ZEZyb250QWxsb3dlZE1ldGhvZHMuQUxMLFxuICAgICAgICAgICAgICAgIGNhY2hlZE1ldGhvZHM6IGNsb3VkZnJvbnQuQ2xvdWRGcm9udEFsbG93ZWRDYWNoZWRNZXRob2RzLkdFVF9IRUFEX09QVElPTlNcbiAgICAgICAgICAgIH0gXVxuICAgIH0gXVxuICB9O1xuXG4gIENsb3VkRnJvbnREaXN0cmlidXRpb25Gb3JBcGlHYXRld2F5KHN0YWNrLCBfYXBpLCBwcm9wcyk7XG5cbiAgZXhwZWN0KHN0YWNrKS50b0hhdmVSZXNvdXJjZUxpa2UoXCJBV1M6OkNsb3VkRnJvbnQ6OkRpc3RyaWJ1dGlvblwiLCB7XG4gICAgRGlzdHJpYnV0aW9uQ29uZmlnOiB7XG4gICAgICAgIERlZmF1bHRDYWNoZUJlaGF2aW9yOiB7XG4gICAgICAgICAgQWxsb3dlZE1ldGhvZHM6IFtcbiAgICAgICAgICAgIFwiREVMRVRFXCIsXG4gICAgICAgICAgICBcIkdFVFwiLFxuICAgICAgICAgICAgXCJIRUFEXCIsXG4gICAgICAgICAgICBcIk9QVElPTlNcIixcbiAgICAgICAgICAgIFwiUEFUQ0hcIixcbiAgICAgICAgICAgIFwiUE9TVFwiLFxuICAgICAgICAgICAgXCJQVVRcIlxuICAgICAgICAgIF0sXG4gICAgICAgICAgQ2FjaGVkTWV0aG9kczogW1xuICAgICAgICAgICAgXCJHRVRcIixcbiAgICAgICAgICAgIFwiSEVBRFwiLFxuICAgICAgICAgICAgXCJPUFRJT05TXCJcbiAgICAgICAgICBdLFxuICAgICAgICAgIENvbXByZXNzOiB0cnVlLFxuICAgICAgICAgIEZvcndhcmRlZFZhbHVlczoge1xuICAgICAgICAgICAgQ29va2llczoge1xuICAgICAgICAgICAgICBGb3J3YXJkOiBcIm5vbmVcIlxuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIFF1ZXJ5U3RyaW5nOiBmYWxzZVxuICAgICAgICAgIH0sXG4gICAgICAgICAgVGFyZ2V0T3JpZ2luSWQ6IFwib3JpZ2luMVwiLFxuICAgICAgICAgIFZpZXdlclByb3RvY29sUG9saWN5OiBcInJlZGlyZWN0LXRvLWh0dHBzXCJcbiAgICAgICAgfSxcbiAgICAgICAgRGVmYXVsdFJvb3RPYmplY3Q6IFwiaW5kZXguaHRtbFwiLFxuICAgICAgICBFbmFibGVkOiB0cnVlLFxuICAgICAgICBIdHRwVmVyc2lvbjogXCJodHRwMlwiLFxuICAgICAgICBJUFY2RW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgTG9nZ2luZzoge1xuICAgICAgICAgIEJ1Y2tldDoge1xuICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgXCJDbG91ZGZyb250TG9nZ2luZ0J1Y2tldDNDM0VGQUE3XCIsXG4gICAgICAgICAgICAgIFwiRG9tYWluTmFtZVwiXG4gICAgICAgICAgICBdXG4gICAgICAgICAgfSxcbiAgICAgICAgICBJbmNsdWRlQ29va2llczogZmFsc2VcbiAgICAgICAgfSxcbiAgICAgICAgT3JpZ2luczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEN1c3RvbU9yaWdpbkNvbmZpZzoge1xuICAgICAgICAgICAgICBIVFRQUG9ydDogODAsXG4gICAgICAgICAgICAgIEhUVFBTUG9ydDogNDQzLFxuICAgICAgICAgICAgICBPcmlnaW5LZWVwYWxpdmVUaW1lb3V0OiA1LFxuICAgICAgICAgICAgICBPcmlnaW5Qcm90b2NvbFBvbGljeTogXCJodHRwLW9ubHlcIixcbiAgICAgICAgICAgICAgT3JpZ2luUmVhZFRpbWVvdXQ6IDMwLFxuICAgICAgICAgICAgICBPcmlnaW5TU0xQcm90b2NvbHM6IFtcbiAgICAgICAgICAgICAgICBcIlRMU3YxLjJcIlxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgRG9tYWluTmFtZToge1xuICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgIFwiaHR0cHM6Ly9cIixcbiAgICAgICAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgICAgICAgUmVmOiBcIlJlc3RBcGkxNDgwQUM0OTlcIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiLmV4ZWN1dGUtYXBpLlwiLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpSZWdpb25cIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiLlwiLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpVUkxTdWZmaXhcIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiL1wiLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFwiUmVzdEFwaTFEZXBsb3ltZW50U3RhZ2Vwcm9kNEZGQzlCQjRcIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiL1wiXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgSWQ6IFwib3JpZ2luMVwiXG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBQcmljZUNsYXNzOiBcIlByaWNlQ2xhc3NfMTAwXCIsXG4gICAgICAgIFZpZXdlckNlcnRpZmljYXRlOiB7XG4gICAgICAgICAgQ2xvdWRGcm9udERlZmF1bHRDZXJ0aWZpY2F0ZTogdHJ1ZVxuICAgICAgICB9XG4gICAgICB9XG4gIH0pO1xuXG59KTsiXX0=