"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
test('snapshot test TableProps default params', () => {
    const stack = new core_1.Stack();
    new dynamodb.Table(stack, 'test-dynamo-defaults', defaults.DefaultTableProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('snapshot test TableWithStream default params', () => {
    const stack = new core_1.Stack();
    new dynamodb.Table(stack, 'test-dynamo-stream-defaults', defaults.DefaultTableWithStreamProps);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test TableProps change billing mode', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        readCapacity: 3,
        writeCapacity: 3,
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        ProvisionedThroughput: {
            ReadCapacityUnits: 3,
            WriteCapacityUnits: 3
        },
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('test TableProps override add sort key', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        },
        sortKey: {
            name: 'sort_key',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            },
            {
                AttributeName: "sort_key",
                KeyType: "RANGE"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            },
            {
                AttributeName: "sort_key",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('test TableWithStreamProps override stream view type', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableWithStreamProps;
    const inProps = {
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        },
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    });
});
test('test buildDynamoDBTable with existingTableObj', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        }
    };
    const existingTableObj = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    defaults.buildDynamoDBTable(stack, {
        existingTableObj
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        ProvisionedThroughput: {
            ReadCapacityUnits: 5,
            WriteCapacityUnits: 5
        }
    }));
});
test('test buildDynamoDBTable without any arguments', () => {
    const stack = new core_1.Stack();
    defaults.buildDynamoDBTable(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        BillingMode: "PAY_PER_REQUEST"
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        SSESpecification: {
            SSEEnabled: true
        }
    }));
});
test('test buildDynamoDBTable with TableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        }
    };
    defaults.buildDynamoDBTable(stack, {
        dynamoTableProps
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        ProvisionedThroughput: {
            ReadCapacityUnits: 5,
            WriteCapacityUnits: 5
        }
    }));
});
test('test buildDynamoDBTableWithStream with TableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    defaults.buildDynamoDBTableWithStream(stack, {
        dynamoTableProps
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    }));
});
test('test buildDynamoDBTableWithStream without any arguments', () => {
    const stack = new core_1.Stack();
    defaults.buildDynamoDBTableWithStream(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        BillingMode: "PAY_PER_REQUEST"
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        SSESpecification: {
            SSEEnabled: true
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_AND_OLD_IMAGES"
        }
    }));
});
test('test buildDynamoDBTableWithStream with existingTableObj', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const existingTableObj = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    defaults.buildDynamoDBTableWithStream(stack, {
        existingTableObj
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    }));
});
//# sourceMappingURL=data:application/json;base64,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