"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const log = require("npmlog");
const process = require("process");
const override_warning_service_1 = require("../lib/override-warning-service");
const __1 = require("../");
// Mock setup
beforeEach(() => {
    jest.resetModules();
    jest.clearAllMocks();
    delete process.env.overrideWarningsEnabled;
});
// --------------------------------------------------------------
// Test override detection: positive, not-nested
// --------------------------------------------------------------
test('Test override detection: positive, not-nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
// --------------------------------------------------------------
// Test override detection: negative, not-nested
// --------------------------------------------------------------
test('Test override detection: negative, not-nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(0);
});
// --------------------------------------------------------------
// Test override detection: positive, nested
// --------------------------------------------------------------
test('Test override detection: positive, nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC',
        keyD: {
            keyDA: 'valueDA',
            keyDB: 'valueDB'
        }
    };
    const b = {
        keyC: 'valueCAnew',
        keyD: {
            keyDA: 'valueDAnew'
        }
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(2);
});
// --------------------------------------------------------------
// Test override detection: negative, nested
// --------------------------------------------------------------
test('Test override detection: negative, nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC',
        keyD: {
            keyDA: 'valueDA',
            keyDB: 'valueDB'
        }
    };
    const b = {
        keyD: {
            keyDA: 'valueDA'
        }
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(0);
});
// --------------------------------------------------------------
// Test override warning on/off: default on
// --------------------------------------------------------------
test('Test override warning on/off: default on', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
// --------------------------------------------------------------
// Test override warning on/off: explicit on
// --------------------------------------------------------------
test('Test override warning on/off: explicit on', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    process.env.overrideWarningsEnabled = 'true';
    // Act
    const warn = jest.spyOn(log, 'warn');
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
// --------------------------------------------------------------
// Test override warning on/off: explicit off
// --------------------------------------------------------------
test('Test override warning on/off: explicit off', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    process.env.overrideWarningsEnabled = 'false';
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(0);
});
// --------------------------------------------------------------
// Test override warning on/off: undefined on
// --------------------------------------------------------------
test('Test override warning on/off: undefined on', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    process.env.overrideWarningsEnabled = undefined;
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
//# sourceMappingURL=data:application/json;base64,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