"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
const aws_s3_1 = require("@aws-cdk/aws-s3");
test('s3 bucket with default params', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {});
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('s3 bucket with default params and bucket names', () => {
    const stack = new core_1.Stack();
    const s3BucketProps = {
        bucketName: 'my-bucket'
    };
    defaults.buildS3Bucket(stack, {
        bucketProps: s3BucketProps
    });
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check exception for Missing existingBucketObj from props for deploy = false', () => {
    const stack = new core_1.Stack();
    try {
        defaults.buildS3Bucket(stack, {});
    }
    catch (e) {
        expect(e).toBeInstanceOf(Error);
    }
});
test('s3 bucket with bucketId', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {}, 'my');
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "myS3LoggingBucketDE461344"
            }
        },
    }));
});
test('s3 bucket with bucketProps', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            bucketName: 'mybucket'
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketName: "mybucket"
    }));
});
test('s3 bucket with life cycle policy', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            lifecycleRules: [{
                    expiration: core_1.Duration.days(365),
                    transitions: [{
                            storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: core_1.Duration.days(30)
                        }, {
                            storageClass: aws_s3_1.StorageClass.GLACIER,
                            transitionAfter: core_1.Duration.days(90)
                        }]
                }]
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LifecycleConfiguration: {
            Rules: [
                {
                    ExpirationInDays: 365,
                    Status: "Enabled",
                    Transitions: [
                        {
                            StorageClass: "STANDARD_IA",
                            TransitionInDays: 30
                        },
                        {
                            StorageClass: "GLACIER",
                            TransitionInDays: 90
                        }
                    ]
                }
            ]
        }
    }));
});
test('s3 bucket with access logging configured', () => {
    const stack = new core_1.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket', {
        serverAccessLogsBucket: new aws_s3_1.Bucket(stack, 'myaccesslogbucket', {})
    });
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsBucket: mybucket
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "mybucket160F8132"
            }
        },
    }));
});
test('Check S3 Bucket policy', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::BucketPolicy", {
        PolicyDocument: {
            Statement: [
                {
                    Sid: 'HttpsOnly',
                    Action: "*",
                    Condition: {
                        Bool: {
                            "aws:SecureTransport": "false"
                        }
                    },
                    Effect: "Deny",
                    Principal: "*",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                {
                                    "Fn::GetAtt": [
                                        "S3Bucket07682993",
                                        "Arn"
                                    ]
                                },
                                "/*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    }));
});
test('s3 bucket with LoggingBucket and versioning turned off', () => {
    const stack = new core_1.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket', {
        serverAccessLogsBucket: new aws_s3_1.Bucket(stack, 'myaccesslogbucket', {})
    });
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsBucket: mybucket,
            serverAccessLogsPrefix: 'access-logs',
            versioned: false
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "mybucket160F8132"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    }));
});
test('s3 bucket versioning turned off', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsPrefix: 'access-logs',
            versioned: false
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "S3LoggingBucket800A2B27"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    }));
});
test('s3 bucket versioning turned on', () => {
    const stack = new core_1.Stack();
    defaults.buildS3Bucket(stack, {
        bucketProps: {
            serverAccessLogsPrefix: 'access-logs',
        }
    });
    assert_1.expect(stack).to(assert_1.haveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        LifecycleConfiguration: {
            Rules: [
                {
                    NoncurrentVersionTransitions: [
                        {
                            StorageClass: "GLACIER",
                            TransitionInDays: 90
                        }
                    ],
                    Status: "Enabled"
                }
            ]
        },
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "S3LoggingBucket800A2B27"
            },
            LogFilePrefix: "access-logs"
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        },
        VersioningConfiguration: {
            Status: "Enabled"
        }
    }));
});
//# sourceMappingURL=data:application/json;base64,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