"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const s3 = require("@aws-cdk/aws-s3");
const kms = require("@aws-cdk/aws-kms");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
const rule_1 = require("@aws-cdk/aws-s3/lib/rule");
test('s3 bucket with default params', () => {
    const stack = new core_1.Stack();
    /** Default Life Cycle policy to transition older versions to Glacier after 90 days */
    const lifecycleRules = [{
            noncurrentVersionTransitions: [{
                    storageClass: rule_1.StorageClass.GLACIER,
                    transitionAfter: core_1.Duration.days(90)
                }]
        }];
    new s3.Bucket(stack, 'test-s3-defaults', defaults.DefaultS3Props(undefined, lifecycleRules));
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('test s3Bucket override versioningConfiguration', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        versioned: false
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-verioning', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        SSEAlgorithm: "AES256"
                    }
                }
            ]
        },
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            BlockPublicPolicy: true,
            IgnorePublicAcls: true,
            RestrictPublicBuckets: true
        }
    });
});
test('test s3Bucket override bucketEncryption', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        encryption: s3.BucketEncryption.KMS,
        encryptionKey: new kms.Key(stack, 'mykeyid')
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-encryption', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        BucketEncryption: {
            ServerSideEncryptionConfiguration: [
                {
                    ServerSideEncryptionByDefault: {
                        KMSMasterKeyID: {
                            "Fn::GetAtt": [
                                "mykeyidFA4203B0",
                                "Arn"
                            ]
                        },
                        SSEAlgorithm: "aws:kms"
                    }
                }
            ]
        },
    });
});
test('test s3Bucket override publicAccessBlockConfiguration', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        blockPublicAccess: s3.BlockPublicAccess.BLOCK_ACLS
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-publicAccessBlock', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        PublicAccessBlockConfiguration: {
            BlockPublicAcls: true,
            IgnorePublicAcls: true
        },
    });
});
test('test s3Bucket add lifecycleConfiguration', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultS3Props();
    const inProps = {
        lifecycleRules: [{
                expiration: core_1.Duration.days(365)
            }]
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new s3.Bucket(stack, 'test-s3-lifecycle', outProps);
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LifecycleConfiguration: {
            Rules: [
                {
                    Status: 'Enabled',
                    ExpirationInDays: 365,
                }
            ]
        }
    });
});
test('test s3Bucket override serverAccessLogsBucket', () => {
    const stack = new core_1.Stack();
    const myLoggingBucket = new s3.Bucket(stack, 'MyS3LoggingBucket', defaults.DefaultS3Props());
    const myS3Props = defaults.DefaultS3Props(myLoggingBucket);
    defaults.buildS3Bucket(stack, {
        bucketProps: myS3Props
    });
    expect(stack).toHaveResource("AWS::S3::Bucket", {
        LoggingConfiguration: {
            DestinationBucketName: {
                Ref: "MyS3LoggingBucket119BE896"
            }
        }
    });
});
//# sourceMappingURL=data:application/json;base64,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