"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const sfn = require("@aws-cdk/aws-stepfunctions");
const aws_logs_1 = require("@aws-cdk/aws-logs");
// --------------------------------------------------------------
// Test minimal deployment with no properties
// --------------------------------------------------------------
test('Test minimal deployment with no properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test deployment w/ custom properties
// --------------------------------------------------------------
test('Test deployment w/ custom properties', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState,
        stateMachineName: 'myStateMachine'
    });
    // Assertion
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        StateMachineName: "myStateMachine"
    });
});
// --------------------------------------------------------------
// Test deployment w/ logging enabled
// --------------------------------------------------------------
test('Test deployment w/ logging enabled', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Log group
    const logGroup = new aws_logs_1.LogGroup(stack, 'myLogGroup', defaults.DefaultLogGroupProps());
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState,
        logs: {
            destination: logGroup,
            level: sfn.LogLevel.FATAL
        }
    });
    // Assertion
    expect(stack).toHaveResource("AWS::StepFunctions::StateMachine", {
        LoggingConfiguration: {
            Destinations: [{
                    CloudWatchLogsLogGroup: {
                        LogGroupArn: {
                            "Fn::GetAtt": [
                                "myLogGroup46524CAB",
                                "Arn"
                            ]
                        }
                    }
                }],
            Level: 'FATAL'
        }
    });
});
// --------------------------------------------------------------
// Check default Cloudwatch perissions
// --------------------------------------------------------------
test('Test deployment w/ logging enabled', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    defaults.buildStateMachine(stack, {
        definition: startState
    });
    // Assertion
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "logs:CreateLogDelivery",
                        "logs:GetLogDelivery",
                        "logs:UpdateLogDelivery",
                        "logs:DeleteLogDelivery",
                        "logs:ListLogDeliveries"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "logs:PutResourcePolicy",
                        "logs:DescribeResourcePolicies",
                        "logs:DescribeLogGroups"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::Join": [
                            "",
                            [
                                "arn:",
                                {
                                    Ref: "AWS::Partition"
                                },
                                ":logs:",
                                {
                                    Ref: "AWS::Region"
                                },
                                ":",
                                {
                                    Ref: "AWS::AccountId"
                                },
                                ":*"
                            ]
                        ]
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
// --------------------------------------------------------------
// Check CW Alarms
// --------------------------------------------------------------
test('Count State Machine CW Alarms', () => {
    // Stack
    const stack = new core_1.Stack();
    // Step function definition
    const startState = new sfn.Pass(stack, 'StartState');
    // Build state machine
    const [sm] = defaults.buildStateMachine(stack, {
        definition: startState
    });
    const cwList = defaults.buildStepFunctionCWAlarms(stack, sm);
    expect(cwList.length).toEqual(3);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3RlcC1mdW5jdGlvbi1oZWxwZXIudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInN0ZXAtZnVuY3Rpb24taGVscGVyLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILFVBQVU7QUFDVix3Q0FBc0M7QUFDdEMsZ0NBQWdDO0FBQ2hDLDRDQUE2QztBQUM3QyxnQ0FBOEI7QUFDOUIsa0RBQWtEO0FBQ2xELGdEQUE2QztBQUU3QyxpRUFBaUU7QUFDakUsNkNBQTZDO0FBQzdDLGlFQUFpRTtBQUNqRSxJQUFJLENBQUMsNENBQTRDLEVBQUUsR0FBRyxFQUFFO0lBQ3BELFFBQVE7SUFDUixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQzFCLDJCQUEyQjtJQUMzQixNQUFNLFVBQVUsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBQ3JELHNCQUFzQjtJQUN0QixRQUFRLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFO1FBQzlCLFVBQVUsRUFBRSxVQUFVO0tBQ3pCLENBQUMsQ0FBQztJQUNILGNBQWM7SUFDZCxNQUFNLENBQUMsbUJBQVUsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLGVBQWUsRUFBRSxDQUFDO0FBQ2pFLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLHVDQUF1QztBQUN2QyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLHNDQUFzQyxFQUFFLEdBQUcsRUFBRTtJQUM5QyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQiwyQkFBMkI7SUFDM0IsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUNyRCxzQkFBc0I7SUFDdEIsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRTtRQUM5QixVQUFVLEVBQUUsVUFBVTtRQUN0QixnQkFBZ0IsRUFBRSxnQkFBZ0I7S0FDckMsQ0FBQyxDQUFDO0lBQ0gsWUFBWTtJQUNaLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0NBQWtDLEVBQUU7UUFDN0QsZ0JBQWdCLEVBQUUsZ0JBQWdCO0tBQ3JDLENBQUMsQ0FBQztBQUNQLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLHFDQUFxQztBQUNyQyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtJQUM1QyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQiwyQkFBMkI7SUFDM0IsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUNyRCxZQUFZO0lBQ1osTUFBTSxRQUFRLEdBQUcsSUFBSSxtQkFBUSxDQUFDLEtBQUssRUFBRSxZQUFZLEVBQUUsUUFBUSxDQUFDLG9CQUFvQixFQUFFLENBQUMsQ0FBQztJQUNwRixzQkFBc0I7SUFDdEIsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRTtRQUM5QixVQUFVLEVBQUUsVUFBVTtRQUN0QixJQUFJLEVBQUU7WUFDRixXQUFXLEVBQUUsUUFBUTtZQUNyQixLQUFLLEVBQUUsR0FBRyxDQUFDLFFBQVEsQ0FBQyxLQUFLO1NBQzVCO0tBQ0osQ0FBQyxDQUFDO0lBQ0gsWUFBWTtJQUNaLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxjQUFjLENBQUMsa0NBQWtDLEVBQUU7UUFDN0Qsb0JBQW9CLEVBQUU7WUFDbEIsWUFBWSxFQUFFLENBQUM7b0JBQ1gsc0JBQXNCLEVBQUU7d0JBQ3RCLFdBQVcsRUFBRTs0QkFDWCxZQUFZLEVBQUU7Z0NBQ1osb0JBQW9CO2dDQUNwQixLQUFLOzZCQUNOO3lCQUNGO3FCQUNGO2lCQUNGLENBQUM7WUFDSixLQUFLLEVBQUUsT0FBTztTQUNqQjtLQUNKLENBQUMsQ0FBQztBQUNQLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLHNDQUFzQztBQUN0QyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLG9DQUFvQyxFQUFFLEdBQUcsRUFBRTtJQUM1QyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQiwyQkFBMkI7SUFDM0IsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUNyRCxzQkFBc0I7SUFDdEIsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRTtRQUM5QixVQUFVLEVBQUUsVUFBVTtLQUN6QixDQUFDLENBQUM7SUFDSCxZQUFZO0lBQ1osTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxrQkFBa0IsRUFBRTtRQUMvQyxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFO3dCQUNOLHdCQUF3Qjt3QkFDeEIscUJBQXFCO3dCQUNyQix3QkFBd0I7d0JBQ3hCLHdCQUF3Qjt3QkFDeEIsd0JBQXdCO3FCQUN6QjtvQkFDRCxNQUFNLEVBQUUsT0FBTztvQkFDZixRQUFRLEVBQUUsR0FBRztpQkFDZDtnQkFDRDtvQkFDRSxNQUFNLEVBQUU7d0JBQ04sd0JBQXdCO3dCQUN4QiwrQkFBK0I7d0JBQy9CLHdCQUF3QjtxQkFDekI7b0JBQ0QsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLFVBQVUsRUFBRTs0QkFDVixFQUFFOzRCQUNGO2dDQUNFLE1BQU07Z0NBQ047b0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjtpQ0FDdEI7Z0NBQ0QsUUFBUTtnQ0FDUjtvQ0FDRSxHQUFHLEVBQUUsYUFBYTtpQ0FDbkI7Z0NBQ0QsR0FBRztnQ0FDSDtvQ0FDRSxHQUFHLEVBQUUsZ0JBQWdCO2lDQUN0QjtnQ0FDRCxJQUFJOzZCQUNMO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7WUFDRCxPQUFPLEVBQUUsWUFBWTtTQUN0QjtLQUNGLENBQUMsQ0FBQztBQUNQLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLGtCQUFrQjtBQUNsQixpRUFBaUU7QUFDakUsSUFBSSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtJQUN6QyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQiwyQkFBMkI7SUFDM0IsTUFBTSxVQUFVLEdBQUcsSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxZQUFZLENBQUMsQ0FBQztJQUNyRCxzQkFBc0I7SUFDdEIsTUFBTSxDQUFDLEVBQUUsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLEVBQUU7UUFDM0MsVUFBVSxFQUFFLFVBQVU7S0FDekIsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxNQUFNLEdBQUcsUUFBUSxDQUFDLHlCQUF5QixDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUU3RCxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQztBQUNuQyxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDIwIEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tIFwiQGF3cy1jZGsvY29yZVwiO1xuaW1wb3J0ICogYXMgZGVmYXVsdHMgZnJvbSAnLi4vJztcbmltcG9ydCB7IFN5bnRoVXRpbHMgfSBmcm9tICdAYXdzLWNkay9hc3NlcnQnO1xuaW1wb3J0ICdAYXdzLWNkay9hc3NlcnQvamVzdCc7XG5pbXBvcnQgKiBhcyBzZm4gZnJvbSAnQGF3cy1jZGsvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0IHsgTG9nR3JvdXAgfSBmcm9tIFwiQGF3cy1jZGsvYXdzLWxvZ3NcIjtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIFRlc3QgbWluaW1hbCBkZXBsb3ltZW50IHdpdGggbm8gcHJvcGVydGllc1xuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoJ1Rlc3QgbWluaW1hbCBkZXBsb3ltZW50IHdpdGggbm8gcHJvcGVydGllcycsICgpID0+IHtcbiAgICAvLyBTdGFja1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgLy8gU3RlcCBmdW5jdGlvbiBkZWZpbml0aW9uXG4gICAgY29uc3Qgc3RhcnRTdGF0ZSA9IG5ldyBzZm4uUGFzcyhzdGFjaywgJ1N0YXJ0U3RhdGUnKTtcbiAgICAvLyBCdWlsZCBzdGF0ZSBtYWNoaW5lXG4gICAgZGVmYXVsdHMuYnVpbGRTdGF0ZU1hY2hpbmUoc3RhY2ssIHtcbiAgICAgICAgZGVmaW5pdGlvbjogc3RhcnRTdGF0ZVxuICAgIH0pO1xuICAgIC8vIEFzc2VydGlvbiAxXG4gICAgZXhwZWN0KFN5bnRoVXRpbHMudG9DbG91ZEZvcm1hdGlvbihzdGFjaykpLnRvTWF0Y2hTbmFwc2hvdCgpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBUZXN0IGRlcGxveW1lbnQgdy8gY3VzdG9tIHByb3BlcnRpZXNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGRlcGxveW1lbnQgdy8gY3VzdG9tIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gICAgLy8gU3RhY2tcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIC8vIFN0ZXAgZnVuY3Rpb24gZGVmaW5pdGlvblxuICAgIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG4gICAgLy8gQnVpbGQgc3RhdGUgbWFjaGluZVxuICAgIGRlZmF1bHRzLmJ1aWxkU3RhdGVNYWNoaW5lKHN0YWNrLCB7XG4gICAgICAgIGRlZmluaXRpb246IHN0YXJ0U3RhdGUsXG4gICAgICAgIHN0YXRlTWFjaGluZU5hbWU6ICdteVN0YXRlTWFjaGluZSdcbiAgICB9KTtcbiAgICAvLyBBc3NlcnRpb25cbiAgICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKFwiQVdTOjpTdGVwRnVuY3Rpb25zOjpTdGF0ZU1hY2hpbmVcIiwge1xuICAgICAgICBTdGF0ZU1hY2hpbmVOYW1lOiBcIm15U3RhdGVNYWNoaW5lXCJcbiAgICB9KTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBkZXBsb3ltZW50IHcvIGxvZ2dpbmcgZW5hYmxlZFxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbnRlc3QoJ1Rlc3QgZGVwbG95bWVudCB3LyBsb2dnaW5nIGVuYWJsZWQnLCAoKSA9PiB7XG4gICAgLy8gU3RhY2tcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIC8vIFN0ZXAgZnVuY3Rpb24gZGVmaW5pdGlvblxuICAgIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG4gICAgLy8gTG9nIGdyb3VwXG4gICAgY29uc3QgbG9nR3JvdXAgPSBuZXcgTG9nR3JvdXAoc3RhY2ssICdteUxvZ0dyb3VwJywgZGVmYXVsdHMuRGVmYXVsdExvZ0dyb3VwUHJvcHMoKSk7XG4gICAgLy8gQnVpbGQgc3RhdGUgbWFjaGluZVxuICAgIGRlZmF1bHRzLmJ1aWxkU3RhdGVNYWNoaW5lKHN0YWNrLCB7XG4gICAgICAgIGRlZmluaXRpb246IHN0YXJ0U3RhdGUsXG4gICAgICAgIGxvZ3M6IHtcbiAgICAgICAgICAgIGRlc3RpbmF0aW9uOiBsb2dHcm91cCxcbiAgICAgICAgICAgIGxldmVsOiBzZm4uTG9nTGV2ZWwuRkFUQUxcbiAgICAgICAgfVxuICAgIH0pO1xuICAgIC8vIEFzc2VydGlvblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoXCJBV1M6OlN0ZXBGdW5jdGlvbnM6OlN0YXRlTWFjaGluZVwiLCB7XG4gICAgICAgIExvZ2dpbmdDb25maWd1cmF0aW9uOiB7XG4gICAgICAgICAgICBEZXN0aW5hdGlvbnM6IFt7XG4gICAgICAgICAgICAgICAgQ2xvdWRXYXRjaExvZ3NMb2dHcm91cDoge1xuICAgICAgICAgICAgICAgICAgTG9nR3JvdXBBcm46IHtcbiAgICAgICAgICAgICAgICAgICAgXCJGbjo6R2V0QXR0XCI6IFtcbiAgICAgICAgICAgICAgICAgICAgICBcIm15TG9nR3JvdXA0NjUyNENBQlwiLFxuICAgICAgICAgICAgICAgICAgICAgIFwiQXJuXCJcbiAgICAgICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgfV0sXG4gICAgICAgICAgICBMZXZlbDogJ0ZBVEFMJ1xuICAgICAgICB9XG4gICAgfSk7XG59KTtcblxuLy8gLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbi8vIENoZWNrIGRlZmF1bHQgQ2xvdWR3YXRjaCBwZXJpc3Npb25zXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBkZXBsb3ltZW50IHcvIGxvZ2dpbmcgZW5hYmxlZCcsICgpID0+IHtcbiAgICAvLyBTdGFja1xuICAgIGNvbnN0IHN0YWNrID0gbmV3IFN0YWNrKCk7XG4gICAgLy8gU3RlcCBmdW5jdGlvbiBkZWZpbml0aW9uXG4gICAgY29uc3Qgc3RhcnRTdGF0ZSA9IG5ldyBzZm4uUGFzcyhzdGFjaywgJ1N0YXJ0U3RhdGUnKTtcbiAgICAvLyBCdWlsZCBzdGF0ZSBtYWNoaW5lXG4gICAgZGVmYXVsdHMuYnVpbGRTdGF0ZU1hY2hpbmUoc3RhY2ssIHtcbiAgICAgICAgZGVmaW5pdGlvbjogc3RhcnRTdGF0ZVxuICAgIH0pO1xuICAgIC8vIEFzc2VydGlvblxuICAgIGV4cGVjdChzdGFjaykudG9IYXZlUmVzb3VyY2UoXCJBV1M6OklBTTo6UG9saWN5XCIsIHtcbiAgICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICAgIFN0YXRlbWVudDogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEFjdGlvbjogW1xuICAgICAgICAgICAgICBcImxvZ3M6Q3JlYXRlTG9nRGVsaXZlcnlcIixcbiAgICAgICAgICAgICAgXCJsb2dzOkdldExvZ0RlbGl2ZXJ5XCIsXG4gICAgICAgICAgICAgIFwibG9nczpVcGRhdGVMb2dEZWxpdmVyeVwiLFxuICAgICAgICAgICAgICBcImxvZ3M6RGVsZXRlTG9nRGVsaXZlcnlcIixcbiAgICAgICAgICAgICAgXCJsb2dzOkxpc3RMb2dEZWxpdmVyaWVzXCJcbiAgICAgICAgICAgIF0sXG4gICAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICAgIFJlc291cmNlOiBcIipcIlxuICAgICAgICAgIH0sXG4gICAgICAgICAge1xuICAgICAgICAgICAgQWN0aW9uOiBbXG4gICAgICAgICAgICAgIFwibG9nczpQdXRSZXNvdXJjZVBvbGljeVwiLFxuICAgICAgICAgICAgICBcImxvZ3M6RGVzY3JpYmVSZXNvdXJjZVBvbGljaWVzXCIsXG4gICAgICAgICAgICAgIFwibG9nczpEZXNjcmliZUxvZ0dyb3Vwc1wiXG4gICAgICAgICAgICBdLFxuICAgICAgICAgICAgRWZmZWN0OiBcIkFsbG93XCIsXG4gICAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgICBcIkZuOjpKb2luXCI6IFtcbiAgICAgICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgICAgIFtcbiAgICAgICAgICAgICAgICAgIFwiYXJuOlwiLFxuICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICBSZWY6IFwiQVdTOjpQYXJ0aXRpb25cIlxuICAgICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICAgIFwiOmxvZ3M6XCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlJlZ2lvblwiXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCI6XCIsXG4gICAgICAgICAgICAgICAgICB7XG4gICAgICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OkFjY291bnRJZFwiXG4gICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgXCI6KlwiXG4gICAgICAgICAgICAgICAgXVxuICAgICAgICAgICAgICBdXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBWZXJzaW9uOiBcIjIwMTItMTAtMTdcIlxuICAgICAgfVxuICAgIH0pO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBDaGVjayBDVyBBbGFybXNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdDb3VudCBTdGF0ZSBNYWNoaW5lIENXIEFsYXJtcycsICgpID0+IHtcbiAgLy8gU3RhY2tcbiAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgLy8gU3RlcCBmdW5jdGlvbiBkZWZpbml0aW9uXG4gIGNvbnN0IHN0YXJ0U3RhdGUgPSBuZXcgc2ZuLlBhc3Moc3RhY2ssICdTdGFydFN0YXRlJyk7XG4gIC8vIEJ1aWxkIHN0YXRlIG1hY2hpbmVcbiAgY29uc3QgW3NtXSA9IGRlZmF1bHRzLmJ1aWxkU3RhdGVNYWNoaW5lKHN0YWNrLCB7XG4gICAgICBkZWZpbml0aW9uOiBzdGFydFN0YXRlXG4gIH0pO1xuICBjb25zdCBjd0xpc3QgPSBkZWZhdWx0cy5idWlsZFN0ZXBGdW5jdGlvbkNXQWxhcm1zKHN0YWNrLCBzbSk7XG5cbiAgZXhwZWN0KGN3TGlzdC5sZW5ndGgpLnRvRXF1YWwoMyk7XG59KTsiXX0=