"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getTypescriptSampleTests = exports.getTypescriptSampleSource = void 0;
/**
 * Return a map of file name to contents for the sample typescript source code
 */
exports.getTypescriptSampleSource = (options) => ({
    "index.ts": ["export * from './api';", "export * from './sample-api';"].join("\n"),
    // This file provides a type-safe interface to the exported OpenApiGatewayLambdaApi construct
    "api.ts": `import { OpenApiGatewayLambdaApi, OpenApiGatewayLambdaApiProps, OpenApiIntegration } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { OperationLookup, OperationConfig } from "${options.typescriptClientPackageName}";
import spec from "../${options.specDir}/${options.parsedSpecFileName}";

export type ApiIntegrations = OperationConfig<OpenApiIntegration>;

export interface ApiProps extends Omit<OpenApiGatewayLambdaApiProps, "spec" | "operationLookup" | "integrations"> {
  readonly integrations: ApiIntegrations;
}

/**
 * Construct for the API Gateway resources defined by the spec
 */
export class Api extends OpenApiGatewayLambdaApi {
  constructor(scope: Construct, id: string, props: ApiProps) {
    super(scope, id, {
      ...props,
      integrations: props.integrations as any,
      spec,
      operationLookup: OperationLookup as any,
    });
  }
}
`,
    ...(options.sampleCode !== false
        ? {
            // Generate an example which instantiates the Api construct
            // TODO: Consider generating this sample from the parsed spec
            "sample-api.ts": `import { Authorizers } from "${options.openApiGatewayPackageName}";
import { Construct } from "constructs";
import { NodejsFunction } from "aws-cdk-lib/aws-lambda-nodejs";
import { Api } from "./api";

/**
 * An example of how to wire lambda handler functions to the API
 */
export class SampleApi extends Api {
  constructor(scope: Construct, id: string) {
    super(scope, id, {
      defaultAuthorizer: Authorizers.iam(),
      integrations: {
        sayHello: {
          function: new NodejsFunction(scope, "say-hello"),
        },
      },
    });
  }
}
`,
            // Generate an example lambda handler
            "sample-api.say-hello.ts": `import { sayHelloHandler, ApiError } from "${options.typescriptClientPackageName}";

/**
 * An example lambda handler which uses the generated handler wrapper to manage marshalling inputs/outputs.
 */
export const handler = sayHelloHandler<ApiError>(async (input) => {
  return {
    statusCode: 200,
    body: {
      message: \`Hello \${input.requestParameters.name}!\`,
    },
  };
});
`,
        }
        : {}),
});
/**
 * Return a map of file name to contents for the sample typescript test code
 */
exports.getTypescriptSampleTests = (options) => ({
    "api.test.ts": `import { Stack } from "aws-cdk-lib";
import { Template } from "aws-cdk-lib/assertions";
import { Code, Function, Runtime } from "aws-cdk-lib/aws-lambda";
import { OperationLookup } from "${options.typescriptClientPackageName}";
import { Api } from "../${options.srcdir}";

/**
 * A simple test to ensure the api construct synthesizes correctly
 */
describe("Api", () => {
  it("should synthesize", () => {
    const stack = new Stack();
    new Api(stack, "ApiTest", {
      // Create a dummy integration for every operation defined in the api
      integrations: Object.fromEntries(Object.keys(OperationLookup).map((operation) => [operation, {
        function: new Function(stack, \`\${operation}Lambda\`, {
          code: Code.fromInline("test"), handler: "test", runtime: Runtime.NODEJS_14_X,
        }),
      }]),
      ) as any,
    });
  
    const template = Template.fromStack(stack);
    expect(template.toJSON()).toMatchSnapshot();
  });
});
`,
});
//# sourceMappingURL=data:application/json;base64,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