"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const constants = require("../../../backend/shared/constants");
const transliterator_lambda_1 = require("../../../backend/transliterator/transliterator.lambda");
jest.mock('child_process');
jest.mock('jsii-docgen');
jest.mock('jsii-rosetta/lib/commands/transliterate');
jest.mock('../../../backend/shared/code-artifact.lambda-shared');
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    transliterator_lambda_1.reset();
    done();
});
describe('VPC Endpoints', () => {
    const previousEnv = process.env;
    const endpoint = 'codeartifact.d.bermuda-triangle-1.amazonaws.com';
    const apiEndpoint = 'codeartifact.api.bermuda-triangle-1.amazonaws.com';
    const domain = 'domain-name';
    const domainOwner = '123456789012';
    beforeAll(() => {
        process.env = {
            ...previousEnv,
            CODE_ARTIFACT_REPOSITORY_ENDPOINT: endpoint,
            CODE_ARTIFACT_DOMAIN_NAME: domain,
            CODE_ARTIFACT_DOMAIN_OWNER: domainOwner,
            CODE_ARTIFACT_API_ENDPOINT: apiEndpoint,
        };
    });
    afterAll(() => {
        process.env = { ...previousEnv };
    });
    test('happy path', async () => {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const forPackage = require('jsii-docgen').Documentation.forPackage;
        forPackage.mockImplementation(async (target) => {
            return new MockDocumentation(target);
        });
        // GIVEN
        const packageScope = 'scope';
        const packageName = 'package-name';
        const packageVersion = '1.2.3-dev.4';
        const event = {
            Records: [{
                    awsRegion: 'bemuda-triangle-1',
                    s3: {
                        bucket: {
                            name: 'dummy-bucket',
                        },
                        object: {
                            key: `${constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                            versionId: 'VersionId',
                        },
                    },
                }],
        };
        const assembly = {
            targets: { python: {} },
        };
        // mock the assembly request
        mockFetchAssembly(assembly);
        // mock the file uploads
        mockPutDocs('/docs-python.md', '/docs-typescript.md');
        const created = await transliterator_lambda_1.handler(event, {});
        expect(created.length).toEqual(2);
        expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-python.md`);
        expect(created[1].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        expect(require('../../../backend/shared/code-artifact.lambda-shared').logInWithCodeArtifact).toHaveBeenCalledWith({
            endpoint,
            domain,
            domainOwner,
            apiEndpoint,
        });
    });
});
test('uploads a file per language (scoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageScope = 'scope';
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const event = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                    },
                    object: {
                        key: `${constants.STORAGE_KEY_PREFIX}%40${packageScope}/${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                        versionId: 'VersionId',
                    },
                },
            }],
    };
    const assembly = {
        targets: { python: {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-python.md', '/docs-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.length).toEqual(2);
    expect(created[0].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-python.md`);
    expect(created[1].key).toEqual(`data/@${packageScope}/${packageName}/v${packageVersion}/docs-typescript.md`);
});
test('uploads a file per submodule (unscoped package)', async () => {
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const forPackage = require('jsii-docgen').Documentation.forPackage;
    forPackage.mockImplementation(async (target) => {
        return new MockDocumentation(target);
    });
    // GIVEN
    const packageName = 'package-name';
    const packageVersion = '1.2.3-dev.4';
    const event = {
        Records: [{
                awsRegion: 'bemuda-triangle-1',
                s3: {
                    bucket: {
                        name: 'dummy-bucket',
                    },
                    object: {
                        key: `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`,
                        versionId: 'VersionId',
                    },
                },
            }],
    };
    const assembly = {
        targets: { python: {} },
        submodules: { '@scope/package-name.sub1': {}, '@scope/package-name.sub2': {} },
    };
    // mock the assembly request
    mockFetchAssembly(assembly);
    // mock the file uploads
    mockPutDocs('/docs-python.md', '/docs-sub1-python.md', '/docs-sub2-python.md', '/docs-typescript.md', '/docs-sub1-typescript.md', '/docs-sub2-typescript.md');
    const created = await transliterator_lambda_1.handler(event, {});
    expect(created.map(({ key }) => key)).toEqual([
        `data/${packageName}/v${packageVersion}/docs-python.md`,
        `data/${packageName}/v${packageVersion}/docs-sub1-python.md`,
        `data/${packageName}/v${packageVersion}/docs-sub2-python.md`,
        `data/${packageName}/v${packageVersion}/docs-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub1-typescript.md`,
        `data/${packageName}/v${packageVersion}/docs-sub2-typescript.md`,
    ]);
});
class MockDocumentation {
    constructor(target) {
        this.target = target;
    }
    render() {
        return {
            render: () => `docs for ${this.target}`,
        };
    }
}
function mockFetchAssembly(response) {
    AWSMock.mock('S3', 'getObject', (request, callback) => {
        if (request.Key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            callback(null, {
                Body: JSON.stringify(response),
            });
        }
        else {
            throw new Error(`Unexpected GET request: ${request.Key}`);
        }
    });
}
function mockPutDocs(...suffixes) {
    AWSMock.mock('S3', 'putObject', (request, callback) => {
        if (suffixes.filter(s => request.Key.endsWith(s)).length > 0) {
            callback(null, { VersionId: `versionId-${request.Key}` });
        }
        else {
            throw new Error(`Unexpected PUT request: ${request.Key}`);
        }
    });
}
//# sourceMappingURL=data:application/json;base64,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