"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const METRICS_NAMESPACE = 'ConstructHub/CatalogBuilder';
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    console.log('Loading existing catalog...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (!data.Body) {
        console.log('Catalog not found. Recreating...');
        const failures = {};
        for await (const object of relevantObjects(BUCKET_NAME)) {
            const docsKey = object.Key;
            try {
                await appendPackage(packages, docsKey, BUCKET_NAME);
            }
            catch (e) {
                failures[docsKey] = e;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setNamespace(METRICS_NAMESPACE);
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric('FailedPackagesOnRecreation', failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    else {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
        console.log('Registering new packages...');
        for (const record of event.Records) {
            // Key names are escaped (`@` as `%40`) in the input payload... Decode it here... We cannot use
            // `decodeURI` here because it does not undo encoding that `encodeURI` would not have done, and
            // that would not replace `@` in the position where it is in the keys... So we have to work on
            // the URI components instead.
            const key = record.s3.object.key.split('/').map((comp) => decodeURIComponent(comp)).join('/');
            // note that we intentionally don't catch errors here to let these
            // event go to the DLQ for manual inspection.
            await appendPackage(packages, key, BUCKET_NAME);
        }
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setNamespace(METRICS_NAMESPACE);
        metrics.putMetric('RegisteredPackagesMajorVersion', catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Upload the result to S3 and exit.
    return aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'text/json',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 */
async function* relevantObjects(bucket) {
    var _a, _b;
    const request = { Bucket: bucket, Prefix: constants.STORAGE_KEY_PREFIX };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.DOCS_KEY_SUFFIX_TYPESCRIPT))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, docsKey, bucketName) {
    var _a, _b, _c;
    console.log(`Processing key: ${docsKey}`);
    const pkgKey = docsKey.replace(constants.DOCS_KEY_SUFFIX_TYPESCRIPT, constants.PACKAGE_KEY_SUFFIX);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    if (found != null && version.compare(found.version) <= 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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