"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const path_1 = require("path");
const url_1 = require("url");
const zlib_1 = require("zlib");
const spec_1 = require("@jsii/spec");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const Environments_1 = require("aws-embedded-metrics/lib/environment/Environments");
const tar_stream_1 = require("tar-stream");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../shared/integrity.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.environmentOverride = Environments_1.default.Lambda;
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // Clear out the default dimensions, we won't need them.
    metrics.setDimensions();
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const result = new Array();
    for (const record of (_a = event.Records) !== null && _a !== void 0 ? _a : []) {
        const payload = JSON.parse(record.body);
        const tarballUri = new url_1.URL(payload.tarballUri);
        if (tarballUri.protocol !== 's3:') {
            throw new Error(`Unsupported protocol in URI: ${tarballUri}`);
        }
        const tarball = await aws.s3().getObject({
            // Note: we drop anything after the first `.` in the host, as we only care about the bucket name.
            Bucket: tarballUri.host.split('.')[0],
            // Note: the pathname part is absolute, so we strip the leading `/`.
            Key: tarballUri.pathname.replace(/^\//, ''),
            VersionId: (_b = tarballUri.searchParams.get('versionId')) !== null && _b !== void 0 ? _b : undefined,
        }).promise();
        const integrityCheck = integrity_lambda_shared_1.integrity(payload, Buffer.from(tarball.Body));
        if (payload.integrity !== integrityCheck) {
            throw new Error(`Integrity check failed: ${payload.integrity} !== ${integrityCheck}`);
        }
        const tar = await gunzip(Buffer.from(tarball.Body));
        const { dotJsii, licenseText, packageJson } = await new Promise((ok, ko) => {
            let dotJsiiBuffer;
            let licenseTextBuffer;
            let packageJsonData;
            const extractor = tar_stream_1.extract({ filenameEncoding: 'utf-8' })
                .once('error', (reason) => {
                ko(reason);
            })
                .once('finish', () => {
                if (dotJsiiBuffer == null) {
                    ko(new Error('No .jsii file found in tarball!'));
                }
                else if (packageJsonData == null) {
                    ko(new Error('No package.json file found in tarball!'));
                }
                else {
                    ok({ dotJsii: dotJsiiBuffer, licenseText: licenseTextBuffer, packageJson: packageJsonData });
                }
            })
                .on('entry', (headers, stream, next) => {
                const chunks = new Array();
                if (headers.name === 'package/.jsii') {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        dotJsiiBuffer = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                else if (headers.name === 'package/package.json') {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        packageJsonData = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                else if (isLicenseFile(headers.name)) {
                    return stream.on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('error', ko)
                        .once('end', () => {
                        licenseTextBuffer = Buffer.concat(chunks);
                        // Skip on next runLoop iteration so we avoid filling the stack.
                        setImmediate(next);
                    })
                        .resume();
                }
                // Skip on next runLoop iteration so we avoid filling the stack.
                return setImmediate(next);
            });
            extractor.write(tar, (err) => {
                if (err != null) {
                    ko(err);
                }
                extractor.end();
            });
        });
        const metadata = { date: payload.time, licenseText: licenseText === null || licenseText === void 0 ? void 0 : licenseText.toString('utf-8') };
        const { license: packageLicense, name: packageName, version: packageVersion } = spec_1.validateAssembly(JSON.parse(dotJsii.toString('utf-8')));
        // Ensure the `.jsii` name, version & license corresponds to those in `package.json`
        const { name: packageJsonName, version: packageJsonVersion, license: packageJsonLicense } = JSON.parse(packageJson.toString('utf-8'));
        if (packageJsonName !== packageName || packageJsonVersion !== packageVersion || packageJsonLicense !== packageLicense) {
            console.log(`Ignoring package with mismatched name, version, and/or license (${packageJsonName}@${packageJsonVersion} is ${packageJsonLicense} !== ${packageName}@${packageVersion} is ${packageLicense})`);
            metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        metrics.putMetric("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, aws_embedded_metrics_1.Unit.Count);
        // Did we identify a license file or not?
        metrics.putMetric("FoundLicenseFile" /* FOUND_LICENSE_FILE */, licenseText != null ? 1 : 0, aws_embedded_metrics_1.Unit.Count);
        const assemblyKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.ASSEMBLY_KEY_SUFFIX}`;
        console.log(`Writing assembly at ${assemblyKey}`);
        const packageKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.PACKAGE_KEY_SUFFIX}`;
        console.log(`Writing package at  ${packageKey}`);
        const metadataKey = `${constants.STORAGE_KEY_PREFIX}${packageName}/v${packageVersion}${constants.METADATA_KEY_SUFFIX}`;
        console.log(`Writing metadata at  ${metadataKey}`);
        // we upload the metadata file first because the catalog builder depends on
        // it and is triggered by the assembly file upload.
        console.log(`${packageName}@${packageVersion} | Uploading package and metadata files`);
        const [pkg, storedMetadata] = await Promise.all([
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: packageKey,
                Body: tarball.Body,
                ContentType: 'application/x-gtar',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
            aws.s3().putObject({
                Bucket: BUCKET_NAME,
                Key: metadataKey,
                Body: JSON.stringify(metadata),
                ContentType: 'application/json',
                Metadata: {
                    'Lambda-Log-Group': context.logGroupName,
                    'Lambda-Log-Stream': context.logStreamName,
                    'Lambda-Run-Id': context.awsRequestId,
                },
            }).promise(),
        ]);
        // now we can upload the assembly.
        console.log(`${packageName}@${packageVersion} | Uploading assembly file`);
        const assembly = await aws.s3().putObject({
            Bucket: BUCKET_NAME,
            Key: assemblyKey,
            Body: dotJsii,
            ContentType: 'application/json',
            Metadata: {
                'Lambda-Log-Group': context.logGroupName,
                'Lambda-Log-Stream': context.logStreamName,
                'Lambda-Run-Id': context.awsRequestId,
            },
        }).promise();
        const created = {
            bucket: BUCKET_NAME,
            assembly: {
                key: assemblyKey,
                versionId: assembly.VersionId,
            },
            package: {
                key: packageKey,
                versionId: pkg.VersionId,
            },
            metadata: {
                key: metadataKey,
                versionId: storedMetadata.VersionId,
            },
        };
        console.log(`Created objects: ${JSON.stringify(created, null, 2)}`);
        result.push(created);
    }
    return result;
});
function gunzip(data) {
    const chunks = new Array();
    return new Promise((ok, ko) => zlib_1.createGunzip()
        .once('error', ko)
        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
        .once('end', () => ok(Buffer.concat(chunks)))
        .end(data));
}
/**
 * Checks whether the provided file name corresponds to a license file or not.
 *
 * @param fileName the file name to be checked.
 *
 * @returns `true` IIF the file is named LICENSE and has the .MD or .TXT
 *          extension, or no extension at all. The test is case-insensitive.
 */
function isLicenseFile(fileName) {
    const ext = path_1.extname(fileName);
    const possibleExtensions = new Set(['', '.md', '.txt']);
    return possibleExtensions.has(ext.toLowerCase())
        && path_1.basename(fileName, ext).toUpperCase() === 'LICENSE';
}
//# sourceMappingURL=data:application/json;base64,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